<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading;

use Enlight_Event_EventManager;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\SubApplicationBuilder\LegacySubApplicationBuilder;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\SubApplicationBuilder\StandaloneSubApplicationBuilder;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\SubApplicationBuilder\SubApplicationBuilder;
use Shopware\Plugins\ViisonCommon\Components\SubApplicationLoading\SubApplicationBuilder\SubApplicationExtensionBuilder;

class LegacyJsLoaderService
{
    const LEGACY_EVENT_NOTIFY_SUB_APPLICATIONS = 'ViisonCommon_SubApplicationRegistration';

    /**
     * @var SubApplicationBuilder[]
     */
    private $subApplicationBuilders = null;

    /**
     * @var Enlight_Event_EventManager
     */
    private $eventManager;

    /**
     * Constructor is required as this is a service.
     */
    public function __construct(Enlight_Event_EventManager $eventManager)
    {
        $this->eventManager = $eventManager;
    }

    /**
     * Registers a sub application with the ViisonCommonJSLoader to automatically load all JavaScript files of the sub
     * application at the appropriate time.
     *
     * @param string $name The name of the sub application to be registered
     * @param string $pluginPath The path to the plugin that this sub application belongs to
     * @param array $dependencies A list of other sub applications that this sub application depends on that need to be loaded together with this sub application
     * @param string $extendsShopwareSubApp A sub application provided in the shopware basis like 'Order', that we want to extend, if applicable
     * @param array $prioritizedFiles Parameter exists only for backward compatibility
     * @param string $controllerClass (optional) The full, namespaced class of the PHP controller that backs the sub application. If not provided "Shopware_Controllers_Backend_$name" is used.
     * @param boolean $disableSmartyRendering (optional) Defaults to false. If set to true, the registered sub application will not be rendered with Smarty, but its files are just added to the view instead.
     * @deprecated This method exists only for backwards compatibility. Use The new SubApplicationRegistrationSubscriber instead.
     */
    public function registerSubApplication(
        $name,
        $pluginPath,
        array $dependencies,
        $extendsShopwareSubApp,
        array $prioritizedFiles = [],
        $controllerClass = null,
        $disableSmartyRendering = false
    ) {
        if (!$disableSmartyRendering) {
            $subApplicationBuilder = new LegacySubApplicationBuilder($name, $pluginPath);
            $subApplicationBuilder->extendingSubApplication($extendsShopwareSubApp);
        } elseif ($extendsShopwareSubApp !== null) {
            $subApplicationBuilder = new SubApplicationExtensionBuilder($name, $pluginPath);
            $subApplicationBuilder->extendingSubApplication($extendsShopwareSubApp);
        } else {
            $subApplicationBuilder = new StandaloneSubApplicationBuilder($name, $pluginPath);
        }

        $subApplicationBuilder->requiringSubApplications($dependencies);
        if ($controllerClass) {
            $subApplicationBuilder->usingCustomControllerClass($controllerClass);
        }

        $this->subApplicationBuilders[] = $subApplicationBuilder;
    }

    /**
     * @return SubApplicationBuilder[]
     */
    public function collectSubApplicationBuilders()
    {
        if ($this->subApplicationBuilders === null) {
            $this->subApplicationBuilders = [];
            $this->eventManager->notify(self::LEGACY_EVENT_NOTIFY_SUB_APPLICATIONS);
        }

        return $this->subApplicationBuilders;
    }
}
