<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

if (!class_exists('ViisonCommon_Plugin_Bootstrap')) {
    require_once('PluginBootstrap.php');
}

/**
 * Version 2 of the common plugin Bootstrap class. This version is based on ViisonCommon_Plugin_Bootstrap,
 * but provides handling of plugin.json files.
 *
 * This class needs to be manually loaded in the respective plugin via:
 *
 * if (!class_exists('ViisonCommon_Plugin_BootstrapV2')) {
 *     require_once('ViisonCommon/PluginBootstrapV2.php');
 * }
 */
class ViisonCommon_Plugin_BootstrapV2 extends ViisonCommon_Plugin_Bootstrap
{
    /**
     * @var array|null $pluginInfo
     */
    protected $pluginInfo;

    /**
     * @return string
     */
    public function getVersion()
    {
        $info = $this->getInfo();

        return $info['version'];
    }

    /**
     * Uses the data provided in the pluginInfo (plugin.json) to create the info
     * returned for this plugin. This excludes 'label' and 'description', which must
     * be added by overriding this method.
     *
     * @return array
     */
    public function getInfo()
    {
        $pluginInfo = $this->getPluginInfo();

        return [
            'version' => $pluginInfo['currentVersion'],
            'autor' => $pluginInfo['author'],
            'copyright' => $pluginInfo['copyright'],
            'license' => $pluginInfo['license'],
            'support' => $pluginInfo['support'],
            'link' => $pluginInfo['link'],
        ];
    }

    /**
     * Tries to find, read and parse a 'plugin.json' file in the plugin's path and,
     * if successful, saves the contents in $this->pluginInfo, which is returned.
     * Calling this method repeatedly will not read/parse the file more than once,
     * because the contents are saved on first call.
     *
     * @return array
     */
    protected function getPluginInfo()
    {
        if ($this->pluginInfo !== null) {
            return $this->pluginInfo;
        }

        $this->pluginInfo = [];

        // Safely determine the plugin path. We cannot use self::Path() here, because only since
        // Shopware 4.3.3 reflection is used to determine the path of this plugin, if none is set
        // in the info. We need to access the plugin path to retrieve the plugin.json file path
        // while loading the plugin info, so we cannot call self::Path() when running on
        // Shopware < 4.3.3. To fix this, we always use reflection to find the file path instead
        // of calling self::Path(). See also:
        //
        // https://github.com/shopware/shopware/commit/385fbd086d27877812a4348b3de8794a86f87014
        //
        $reflection = new \ReflectionClass($this);
        $pluginPath = dirname($reflection->getFileName()) . DIRECTORY_SEPARATOR;

        // Try to find a plugin.json file
        $filePath = $pluginPath . 'plugin.json';
        if (!file_exists($filePath)) {
            // plugin.json not found
            return $this->pluginInfo;
        }

        // Read and parse the plugin.json file
        $fileContent = file_get_contents($filePath);
        if ($fileContent === false) {
            // Failed to read plugin.json
            return $this->pluginInfo;
        }
        $pluginInfo = json_decode($fileContent, true);
        if (json_last_error() !== JSON_ERROR_NONE || empty($pluginInfo)) {
            // Failed to parse plugin.json
            return $this->pluginInfo;
        }
        $this->pluginInfo = $pluginInfo;

        return $this->pluginInfo;
    }
}
