<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonUPS\Classes\Exceptions;

use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;
use SimpleXMLElement;

class UPSCommunicationException extends \Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @param string $message
     * @param string $snippetName
     * @param mixed[] $snippetArguments
     */
    public function __construct($message, $snippetName, array $snippetArguments = [])
    {
        parent::__construct($message);
        $this->setSnippetName($snippetName);
        $this->setSnippetNamespace('exceptions/viison_ups/communication');
        $this->setSnippetArguments($snippetArguments);
    }

    /**
     * @param $requestedUrl
     * @return UPSCommunicationException
     */
    public static function webserviceOffline($requestedUrl)
    {
        return new self(
            sprintf(
                'The resource at the URL "%s" could not be loaded. Either UPS Web Service is currently unavailable ' .
                'or the server settings do not allow connection to UPS Web Service. Please try again. If this ' .
                'problem persists, check the firewall settings and the PHP configuration ("php. ini") of the server ' .
                'if necessary. For example, the option "allow_url_fopen" must be activated ("On").',
                $requestedUrl
            ),
            'webserviceOffline',
            [
                $requestedUrl
            ]
        );
    }

    /**
     * @param SimpleXMLElement $responseXML
     * @return UPSCommunicationException
     */
    public static function requestFailed(SimpleXMLElement $responseXML)
    {
        $errorDescription = $responseXML->Response->Error->ErrorDescription;

        return new self(
            sprintf('The request could not be processed by the UPS server: %s', $errorDescription),
            'requestFailed',
            [
                $errorDescription
            ]
        );
    }

    /**
     * @param int $productId
     * @return UPSCommunicationException
     */
    public static function productNotFound($productId)
    {
        return new self(
            sprintf(
                'Selected UPS product (id=%d) does not exist. Please refer to the UPS Adapter documentation ' .
                'and check the shipping method of your order for a correctly assigned UPS product.',
                $productId
            ),
            'productNotFound',
            [
                $productId
            ]
        );
    }

    /**
     * @param int $orderId
     * @return UPSCommunicationException
     */
    public static function noValidProductFound($orderId)
    {
        return new self(
            sprintf(
                'No valid UPS product found for order (id=%d). Please refer to the UPS Adapter documentation ' .
                'and check the shipping method of your order for a correctly assigned UPS product.',
                $orderId
            ),
            'noValidProductFound',
            [
                $orderId
            ]
        );
    }

    /**
     * @return UPSCommunicationException
     */
    public static function accessPointShipmentRequiresOrder()
    {
        return new self(
            'Unfortunately, deliveries to AccessPoints are currently supported only for shipments that are assigned to an order.',
            'accessPointShipmentRequiresOrder'
        );
    }
}
