<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonUPS\Classes;

use Shopware\Plugins\ViisonCommon\Classes\Util\Util;
use Shopware\Plugins\ViisonShippingCommon\Classes\PluginInfo as ShippingCommonPluginInfo;

/**
 * This class provides plugin specific information like for example database table
 * names. Instances of this class are injected into generalized classes of the
 * ShippingCommon module.
 */
class PluginInfo extends ShippingCommonPluginInfo
{
    /**
     * Returns the table name of the plugin specific order table
     * @return string
     */
    public function getOrderTableName()
    {
        return 's_order_viison_ups';
    }

    /**
     * Returns the table name of the plugin specific config table
     * @return string
     */
    public function getConfigTableName()
    {
        return 's_core_shops_viison_ups';
    }

    /**
     * Returns the table name of the plugin specific product table
     * @return string
     */
    public function getProductTableName()
    {
        return 's_viison_ups_products';
    }

    /**
     * Returns the table name of the plugin specific product mapping table
     * @return string
     */
    public function getProductMappingTableName()
    {
        return 's_premium_dispatch_viison_ups';
    }

    /**
     * Returns the table name of the plugin specific packaging type table
     * @return string
     */
    public function getPackagingTypeTableName()
    {
        return 's_viison_ups_packaging_codes';
    }

    /**
     * Returns the file prefix to be used for all plugin specific documents (labels, return labels etc.)
     * stored in the shopware documents directory
     * @return string
     */
    public function getFilePrefix()
    {
        return 'ups';
    }

    /**
     * Returns the dispatch method name to be displayed to the user
     * @return string
     */
    public function getPluginDisplayName()
    {
        return 'UPS Versand';
    }

    /**
     * Returns the plugin name also used as the prefix for all classes (e.g. 'ViisonIntraship')
     * @return string
     */
    public function getPluginName()
    {
        return 'ViisonUPS';
    }

    /**
     * Returns the return label email template name
     * @return string
     */
    public function getReturnLabelEmailTemplateName()
    {
        return 'ViisonUPSReturnLabelEmail';
    }

    /**
     * Returns the URL that can be used to track shipments. When tracking a shipment, the respective tracking
     * code of the shipment is appended to this URL.
     * @return string
     */
    public function getTrackingURL($activeLanguage = 'de')
    {
        $localeParams = ($activeLanguage == 'de' ? 'loc=de_DE&' : '');

        return sprintf('http://www.ups.com/WebTracking/track?%strackNums=', $localeParams);
    }

    /**
     * Returns the tracking URL in the language of the current backend session.
     * @return string
     */
    public function getTrackingURLForActiveLocale()
    {
        return $this->getTrackingURL(Shopware()->Container()->get('locale')->getLanguage());
    }

    /**
     * Returns the delimiter to be used to separate individual tracking numbers in the parameter send to the
     * tracking URL if multiple shipments are tracked at the same time.
     * @return string
     */
    public function getTrackingNumberDelimiter()
    {
        return urlencode("\r\n");
    }

    /**
     * Returns the maximum number of labels per order.
     * @return int
     */
    public function getMaxLabels()
    {
        if (Util::assertMinimumShopwareVersion('5.5.4')) {
            return PHP_INT_MAX;
        }

        return 13;
    }

}
