<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonUPS\Classes;

use Shopware\Plugins\ViisonShippingCommon\Classes\ShippingLabelCreationResult;
use Shopware\Plugins\ViisonShippingCommon\Classes\ShippingLabelGenerator as ShippingCommonLabelGenerator;
use Shopware\Plugins\ViisonUPS\Util;
use Shopware\Plugins\ViisonUPS\Classes\ShippingUtil;
use Shopware\Plugins\ViisonUPS\Classes\UPSCommunication;

/**
 * Implementation of the ShippingLabelGenerator interface of the ShippingCommon module. Delegates
 * label creation and deletion to the UPSCommunication class.
 */
class ShippingLabelGenerator extends ShippingCommonLabelGenerator
{

    /**
     * @var UPSCommunication
     */
    private $com;

    public function __construct()
    {
        parent::__construct(Util::instance(), new ShippingUtil());
        $this->com = new UPSCommunication();
    }

    /**
     * Create a new shipping label for a given order.
     *
     * @param int $orderId The id of the order for which a label should be created.
     * @param float $weight The weight to use as the shipment's weight.
     * @param bool $useItemWeights A flag indicating whether to use the weight defined in each item or to calculate the item weights from the total weight. This parameter is only relevant for shipments containing an export document (e.g. Weltpaket).
     * @param array $shippingDetails An optional array containing the receivers address data.
     * @param array $packageDimensions An optional array containing the dimensions of the package, for which a new label shall be created.
     * @param array $settings An optional array containing the settings for the new label.
     * @param array $extraSettings An optional array containing plugin-specific settings
     * @param array $exportDocumentItems An optional array containing the order items and quantities, which will be contained in the shipment.
     * @return ShippingLabelCreationResult
     */
    public function createLabel($orderId, $weight, $useItemWeights, $shippingDetails, $packageDimensions, $settings, $extraSettings, $exportDocumentItems)
    {
        if (is_array($packageDimensions)
            && is_null($packageDimensions['length'])
            && is_null($packageDimensions['width'])
            && is_null($packageDimensions['height'])) {
            $packageDimensions = null;
        }
        return $this->com->createLabel($orderId, $weight, false, $useItemWeights, $shippingDetails, $packageDimensions, $settings, $extraSettings);
    }

    /**
     * Create a new return label for a given order.
     *
     * @param int $orderId The id of the order for which a label should be created.
     * @param float $weight The weight to use as the shipment's weight.
     * @param array $shippingDetails
     * @params array|null $packagDimensions
     * @params array|null $settings
     * @param array $extraSettings An optional array containing plugin-specific settings
     * @return string The tracking code of the newly created label.
     */
    public function createReturnLabel($orderId, $weight, $shippingDetails, $packageDimensions, $settings, $extraSettings)
    {
        $result = $this->com->createLabel($orderId, $weight, true, false, $shippingDetails);
        $trackingCode = $result->getNewTrackingCode();
        return $trackingCode;
    }

    /**
     * Deletes the label with given tracking code.
     *
     * @param string $trackingCode
     * @return void
     */
    public function deleteLabel($trackingCode)
    {
        $this->com->deleteLabel($trackingCode);
    }

    /**
     * Deletes the return label with given tracking code.
     *
     * @param string $trackingCode
     * @return void
     */
    public function deleteReturnLabel($trackingCode)
    {
        $this->deleteLabel($trackingCode);
    }
}
