<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonUPS\Classes;

use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonShippingCommon\Classes\ShippingDocumentType;
use Shopware\Plugins\ViisonShippingCommon\Classes\ShippingProvider as ShippingCommonProvider;
use Shopware\Plugins\ViisonShippingCommon\Structs\PageSize;
use Shopware\Plugins\ViisonUPS\Util;

// Make sure all classes are included if this file is invoked directly via the API
include_once(__DIR__ . '/../Bootstrap.php');

class ShippingProvider extends ShippingCommonProvider
{
    public function __construct()
    {
        $shippingLabelGenerator = new ShippingLabelGenerator();
        $shippingUtil = new ShippingUtil();
        parent::__construct(Util::instance(), $shippingLabelGenerator, $shippingUtil);
    }

    /**
     * @inheritdoc
     */
    public function getGdprMailConfiguration(Shop $shop)
    {
        return $this->util->config(
            $shop->getId(),
            'gdprMailConfiguration'
        );
    }

    /**
     * @inheritdoc
     */
    public function getGdprPhoneConfiguration(Shop $shop)
    {
        return $this->util->config(
            $shop->getId(),
            'gdprPhoneConfiguration'
        );
    }

    /**
     * @inheritdoc
     */
    public function getAvailableShippingDocumentTypes()
    {
        /** @var \Enlight_Components_Snippet_Manager $snippets */
        $snippets = Shopware()->Container()->get('snippets');
        $namespace = $snippets->getNamespace('backend/viison_ups_document_types/main');
        $documentTypeName = $namespace->get('shipping_label/name');

        return [
            new ShippingDocumentType(
                UPSShippingDocument::DOCUMENT_TYPE_ID_A4,
                $documentTypeName,
                new PageSize('A4', 210, 297)
            ),
            new ShippingDocumentType(
                UPSShippingDocument::DOCUMENT_TYPE_ID_A5,
                $documentTypeName,
                new PageSize('A5', 148, 210)
            ),
            new ShippingDocumentType(
                UPSShippingDocument::DOCUMENT_TYPE_ID_4X7_INCH,
                $documentTypeName,
                new PageSize('4"x7"', UPSLabelImage::LABEL_WIDTH_IN_MM, UPSLabelImage::LABEL_HEIGHT_7_INCH_IN_MM)
            ),
            new ShippingDocumentType(
                UPSShippingDocument::DOCUMENT_TYPE_ID_4X6_INCH,
                $documentTypeName,
                new PageSize('4"x6"', UPSLabelImage::LABEL_WIDTH_IN_MM, UPSLabelImage::LABEL_HEIGHT_6_INCH_IN_MM)
            ),
        ];
    }

    /**
     * @inheritdoc
     */
    public function getAllDocumentDescriptions($orderId, $undispatchedOnly = false)
    {
        $documents = parent::getAllDocumentDescriptions($orderId, $undispatchedOnly);

        return array_map(function ($document) {
            return UPSShippingDocument::fromShippingDocument($document);
        }, $documents);
    }
}
