<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonUPS\Classes;

use Shopware\Plugins\ViisonShippingCommon\Classes\ShippingDocument;

include_once(__DIR__ . '/../ViisonShippingCommon/ShippingProviderInterfaceLoader.php');

class UPSShippingDocument implements
    \ViisonPickwareConnector_API_ShippingDocument,
    \ViisonPickwareMobile_Interfaces_ShippingProvider_ShippingDocument
{
    const DOCUMENT_TYPE_ID_A4 = 1;
    const DOCUMENT_TYPE_ID_A5 = 2;
    const DOCUMENT_TYPE_ID_4X7_INCH = 3;
    const DOCUMENT_TYPE_ID_4X6_INCH = 4;

    const PAGE_SIZE_A4 = 'A4';
    const PAGE_SIZE_A5 = 'A5';
    const PAGE_SIZE_4X6_INCH = '4x6-inch';
    const PAGE_SIZE_4X7_INCH = '4x7-inch';

    /**
     * @var string
     */
    private $type;

    /**
     * @var string
     */
    private $trackingCode;

    /**
     * @var string
     */
    private $pageSize;

    /**
     * @var int
     */
    private $documentTypeId;

    /**
     * @param string $type The type of this document, e.g. 'exportDocument'.
     * @param string $trackingCode The tracking code of this document.
     * @param string $pageSize The page size of the real document, which is represented by this instance.
     */
    public function __construct($type, $trackingCode, $pageSize)
    {
        $this->type = $type;
        $this->trackingCode = $trackingCode;
        $this->pageSize = $pageSize;

        switch ($this->pageSize) {
            case self::PAGE_SIZE_4X6_INCH:
                $this->documentTypeId = self::DOCUMENT_TYPE_ID_4X6_INCH;
                break;
            case self::PAGE_SIZE_4X7_INCH:
                $this->documentTypeId = self::DOCUMENT_TYPE_ID_4X7_INCH;
                break;

            case self::PAGE_SIZE_A4:
                $this->documentTypeId = self::DOCUMENT_TYPE_ID_A4;
                break;

            case self::PAGE_SIZE_A5:
                $this->documentTypeId = self::DOCUMENT_TYPE_ID_A5;
                break;

            default:
                throw new \RuntimeException(sprintf('Paper size %s of document is unknown.', $this->pageSize));
        }
    }

    /**
     * Converts a ShippingDocument into a UPSShippingDocument
     *
     * @param ShippingDocument $document
     * @return UPSShippingDocument
     */
    public static function fromShippingDocument(ShippingDocument $document)
    {
        return new self(
            $document->getType(),
            $document->getTrackingCode(),
            $document->getPageSize()
        );
    }

    /**
     * @inheritdoc
     */
    public function getPageSize()
    {
        // Since the pageSize is used to determine a fallback printer within the app and since the app
        // only supports fallback printer for the formats A4, A5, A6, we can't return the inch sizes here.
        if ($this->pageSize !== self::PAGE_SIZE_A4) {
            return 'A5';
        }

        return $this->pageSize;
    }

    /**
     * @inheritdoc
     */
    public function getDocumentTypeId()
    {
        return $this->documentTypeId;
    }

    /**
     * @inheritDoc
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @inheritDoc
     */
    public function getIdentifier()
    {
        return $this->getType() . PluginInfo::LABEL_IDENTIFIER_SEPARATOR . $this->getTrackingCode();
    }

    /**
     * @inheritDoc
     */
    public function getTrackingCode()
    {
        return $this->trackingCode;
    }
}
