<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Plugins\ViisonShippingCommon\Controllers\Backend\ViisonShippingCommonOrder;
use Shopware\Plugins\ViisonUPS\Classes\ShippingLabelGenerator;
use Shopware\Plugins\ViisonUPS\Classes\ShippingUtil;
use Shopware\Plugins\ViisonUPS\Classes\ShippingProvider;
use Shopware\Plugins\ViisonUPS\Util;

/**
 * This controller adds some actions called in the order backend controller.
 * These actions are responsible for creating and deleting shipping labels via UPS
 * as well as returning the pre calculated weight of shipments or multiple shipping labels
 * merged in one PDF file.
 */
class Shopware_Controllers_Backend_ViisonUPSOrder extends ViisonShippingCommonOrder
{
    public function __construct(Enlight_Controller_Request_Request $request = null, Enlight_Controller_Response_Response $response = null)
    {
        $shippingLabelGenerator = new ShippingLabelGenerator();
        $shippingProvider = new ShippingProvider();
        $shippingUtil = new ShippingUtil();
        parent::__construct($request, $response, $shippingProvider, Util::instance(), $shippingLabelGenerator, $shippingUtil);
    }

    /**
     * Returns the default packaging type id according to the configuration
     * of the shop that the order was placed in.
     */
    public function getDefaultPackagingTypeAction()
    {
        $this->View()->success = true;

        // Get order id
        $orderId = $this->Request()->getParam('orderId');
        if (!isset($orderId)) {
            $this->View()->success = false;
            $this->View()->message = 'Missing orderId.';
            return;
        }

        // Determine whether the package dimensions are required
        $this->View()->defaultPackagingType = $this->util->getDefaultPackagingType($orderId);
    }

    /**
     * Returns the residential address indicator for the given order.
     */
    public function getResidentialAddressAction()
    {
        $this->View()->success = true;

        // Get order id
        $orderId = $this->Request()->getParam('orderId');
        if (!isset($orderId)) {
            $this->View()->success = false;
            $this->View()->message = 'Missing orderId.';
            return;
        }

        $this->View()->residentialAddress = $this->util->getResidentialAddress($orderId);
    }

    protected function getShippingDetails($request)
    {
        $details = parent::getShippingDetails($request);
        if (is_null($details)) {
            return null;
        }
        $details['residentialaddress'] = $request->getParam('detailsResidentialAddress');
        return $details;
    }

    /* Override */
    protected function getShippingSettings($request)
    {
        $settings = parent::getShippingSettings($request);
        if (is_null($settings)) {
            return null;
        }

        $settings['packagingtype'] = $request->getParam('settingsPackagingType', null);
        $settings['higherinsurance'] = $request->getParam('settingsHigherInsurance', null);
        $settings['additionalhandlingrequired'] = $request->getParam('settingsAdditionalHandlingRequired', null);
        $settings['insuredvalue'] = $request->getParam('settingsInsuredValue', null);
        $settings['insuredvaluecurrency'] = $request->getParam('settingsInsuredValueCurrency', null);
        $settings['issaturdaydelivery'] = $request->getParam('settingsSaturdayDelivery', null);

        return $settings;
    }

    /**
     * @Override
     * @return array Data that can be used inside the Views
     */
    public function getViewParams()
    {
        return array_merge(
            parent::getViewParams(),
            array('dimensionConstraints' => Shopware\Plugins\ViisonUPS\Classes\ShipConfirmRequest::getDimensionConstraints())
        );
    }
}
