<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonUPS;

use Shopware\Plugins\ViisonShippingCommon\Util as ShippingCommonUtil;
use Shopware\Plugins\ViisonUPS\Classes\PluginInfo;

/**
 * Contains static utility methods and values which are used by several classes of the ViisonUPS plugin.
 */
class Util extends ShippingCommonUtil
{
    /**
     * @inheritdoc
     */
    protected function getKey()
    {
        return null;
    }

    /**
     * Returns an array containing all config fields that are always needed for label creation.
     *
     * @override
     * @return array
     */
    protected function getMandatoryConfigFields()
    {
        return array('senderName', 'streetName', 'streetNumber', 'zipCode', 'city', 'phoneNumber', 'userId', 'password', 'accessKey');
    }

    /**
     * Returns if the config contains all fields necessary to create labels for the given order.
     *
     * @override
     * @param int $orderId
     * @param array $config
     * @return bool
     */
    protected function hasOrderSpecificValidConfig($orderId, $config)
    {
        return true;
    }

    /**
     * Protected constructor to prevent creating a new instance of the
     * Singleton via the `new` operator from outside of this class.
     */
    protected function __construct()
    {
        parent::__construct(new PluginInfo);
    }

    /**
     * Private clone method to prevent cloning of the instance of the
     * Singleton instance.
     *
     * @return void
     */
    private function __clone()
    {
    }

    /**
     * Private unserialize method to prevent unserializing of the Singleton
     * instance.
     *
     * @return void
     */
    public function __wakeup()
    {
    }

    public static function instance()
    {
        static $instance = null;
        if (null === $instance) {
            $instance = new static();
        }
        return $instance;
    }

    /**
     * Returns an array containing the plugin specific product fields and their respective values for the given product table row.
     *
     * @override
     * @param int $shopId
     * @param array $product
     * @return array
     */
    protected function getExtraProductFields($shopId, $product)
    {
        return array();
    }

    /**
     * @Inheritdoc
     */
    public function getDefaultShipmentWeight($shopId, $product)
    {
        return $this->getDefaultShipmentWeightWithoutProduct($shopId);
    }

    /**
     * @Inheritdoc
     */
    public function getDefaultShipmentWeightWithoutProduct($shopId)
    {
        return $this->config($shopId, 'defaultShipmentWeight');
    }

    /**
     * Returns the default packaging type for order with the given id.
     * @return int|null
     */
    public function getDefaultPackagingType($orderId)
    {
        $shopId = $this->originatingShop($orderId);

        $configTable = new \Zend_Db_Table($this->pluginInfo->getConfigTableName());

        // Load the default packaging type from the configuration of the current shop
        $defaultPackagingType = Shopware()->Db()->fetchOne(
            $configTable->select()
            ->from($configTable, array('defaultPackagingType'))
            ->where('shopId = ?', $shopId)
        );

        // Default packaging type should be either null if not configured or a number
        if (!is_null($defaultPackagingType)) {
            $defaultPackagingType = intval($defaultPackagingType);
        }

        return $defaultPackagingType;
    }

    /**
     * Returns the residential address indicator for the order with the given id.
     * @return bool|null
     */
    public function getResidentialAddress($orderId)
    {
        $order = $this->getOrder($orderId);
        if (is_null($order)) {
            return null;
        }
        return empty($order['company']);
    }
}
