// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * This controller implements the application logic of the UPS tab that
 * is part of the order detail window. It also handles the event that
 * is fired when clicking the 'Add label' icon in the order list view to
 * directly access the UPS tab.
 */
//{namespace name="backend/viison_ups_order/order"}
Ext.define('Shopware.apps.ViisonUPSOrder.controller.Tab', {

    /**
     * Extend the order tab controller from the ShippingCommon module
     */
    extend: 'Shopware.apps.ViisonShippingCommonOrder.controller.Tab',

    /**
     * The ViisonUPSShipping module is required to use the product store
     */
    requires: 'Shopware.apps.ViisonUPSShipping',

    /**
     * Plugin-specific properties required by the parent class
     */
    dispatchServiceProviderPrefix: 'ViisonUPS',
    dispatchServiceProviderPanelClass: 'order-viison-ups-tab',
    productStoreClass: 'Shopware.apps.ViisonUPSShipping.store.Product',
    labelConfirmPanelClass: 'Shopware.apps.ViisonUPSOrder.view.detail.LabelConfirm',
    shippingLabelClass: 'Shopware.apps.ViisonUPSOrder.model.ShippingLabel',
    sendMailURL: '{url controller="ViisonUPSOrder" action="sendMail"}',
    downloadDocumentsURL: '{url controller="ViisonUPSOrder" action="downloadDocuments"}',
    calculateShippingWeightURL: '{url controller="ViisonUPSOrder" action="calculateShippingWeight"}',
    getShippingOrderDataURL: '{url controller="ViisonUPSOrder" action="getShippingOrderData"}',
    dispatchServiceProviderCSSClass: Ext.baseCSSPrefix + 'ups',
    dispatchServiceProviderLabelConfirmPanel: 'order-viison-ups-label-confirm-panel',
    listDispatchActionsControllerClass: 'Shopware.apps.ViisonUPSOrder.controller.ListDispatchActions',
    orderConfigDataClass: 'Shopware.apps.ViisonUPSOrder.store.OrderConfigData',

    /**
     * The maximum number of shipping labels per order.
     */
    maxNumberOfLabels: {$pluginInfo->getMaxLabels()},

    /**
     * @Override
     *
     * Additional properties required by the parent class.
     */
    mergedLabelsFromOrderActionUrl: '{url controller="ViisonUPSOrder" action="getMergedLabelsForOrder"}',

    customSnippets: {
        notifications: {
            growlMessage: '{s name="detail/notifications/growlMessage"}{/s}'
        }
    },

    /**
     * Loads the default packaging type for the given order from the backend
     * controller. This information is saved in the record. If already a label
     * confirm window and panel exist, it will be updated.
     *
     * @param record The record representing the order.
     */
    updatePackagingType: function(record) {
        var me = this;
        Ext.Ajax.request({
            url: '{url controller="ViisonUPSOrder" action="getDefaultPackagingType"}',
            params: {
                orderId: record.getId()
            },
            success: function(response) {
                var data = Ext.JSON.decode(response.responseText);
                record.ViisonUPS = record.ViisonUPS || {};
                record.ViisonUPS.packagingType = data.defaultPackagingType;
                if (me.getLabelConfirmWindow() && me.getLabelConfirmWindow().dataPanel) {
                    // Force update of displayed values in label confirm view
                    me.getLabelConfirmWindow().dataPanel.setValues();
                }
            }
        });
    },

    /**
     * Loads the residential address indicator for the given order from the backend
     * controller. This information is saved in the record. If already a label
     * confirm window and panel exist, it will be updated.
     *
     * @param record The record representing the order.
     */
    updateResidentialAddress: function(record) {
        var me = this;
        Ext.Ajax.request({
            url: '{url controller="ViisonUPSOrder" action="getResidentialAddress"}',
            params: {
                orderId: record.getId()
            },
            success: function(response) {
                var data = Ext.JSON.decode(response.responseText);
                record.ViisonUPS = record.ViisonUPS || {};
                record.ViisonUPS.residentialAddress = data.residentialAddress;
                if (me.getLabelConfirmWindow() && me.getLabelConfirmWindow().dataPanel) {
                    // Force update of displayed values in label confirm view
                    me.getLabelConfirmWindow().dataPanel.setValues();
                }
            }
        });
    },

    /**
     * Saves plugin-specific order attributes in the order record.
     *
     * @param orderRecord
     */
    loadDataForOrder: function(orderRecord) {
        var me = this;

        me.callParent(arguments);

        // Load the default packaging type and residential address indicator for this order
        this.updatePackagingType(orderRecord);
        this.updateResidentialAddress(orderRecord);
    }

});
