<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Controllers\Backend;

if (!class_exists('Shopware_Controllers_Backend_Analytics')) {
    require_once('Shopware_Controllers_Backend_Analytics');
}

/**
 * Adds some basic parameter getters that are used by the Analytics Controller of different plugins.
 */
abstract class AbstractAnalyticsController extends \Shopware_Controllers_Backend_Analytics
{

    /**
     * Returns "start" parameter from the request.
     * Use default null if no param was provided
     *
     * @return null|int
     */
    protected function getStart()
    {
        $start = $this->Request()->getParam('start');

        return $start ? (int) $start : null;
    }

    /**
     * Returns "limit" parameter from the request.
     * Set limit default to null to ensure CSV export contains all entries if no limit was given.
     *
     * @return null|int
     */
    protected function getLimit()
    {
        $limit = $this->Request()->getParam('limit');

        return $limit ? (int) $limit : null;
    }

    /**
     * Returns "fromDate" parameter from the request.
     * Create from-DateTime object with time 00:00:00, or return null if no param was given.
     *
     * @return \DateTime|null
     */
    protected function getFromDate()
    {
        $fromDate = $this->Request()->getParam('fromDate');

        return $fromDate ? new \DateTime($fromDate) : null;
    }

    /**
     * Returns "toDate" parameter from the request.
     * Create to-DateTime with time 23:59:59, or return null if no param was given.
     *
     * @return \DateTime|null
     */
    protected function getToDate()
    {
        $toDate = $this->Request()->getParam('toDate');
        if ($toDate) {
            $toDate = new \DateTime($toDate);
            $toDate = $toDate->add(new \DateInterval('P1D'));
            $toDate = $toDate->sub(new \DateInterval('PT1S'));
        }

        return $toDate ? $toDate : null;
    }

    /**
     * Returns "selectedShops" parameter from the request as an array of IDs or null if no param was given.
     *
     * @return int[]|null
     */
    protected function getSelectedShopIds()
    {
        $selectedShopIds = $this->Request()->getParam('selectedShops');

        return $selectedShopIds ? array_map('intval', explode(',', trim($selectedShopIds))) : null;
    }

    /**
     * Returns "selectedCategories" parameter from the request as an array of IDs or null if no param was given.
     *
     * @return int[]|null
     */
    protected function getSelectedCategoryIds()
    {

        $selectedCategoryIds = $this->Request()->getParam('selectedCategories');

        return $selectedCategoryIds ? array_map('intval', explode(',', trim($selectedCategoryIds))) : null;
    }

    /**
     * Returns "selectedCustomerGroups" parameter from the request as an array of IDs or null if no param was given.
     *
     * @return int[]|null
     */
    protected function getSelectedCustomerGroupIds()
    {

        $selectedCustomerGroupIds = $this->Request()->getParam('selectedCustomerGroups');

        return $selectedCustomerGroupIds ? array_map('intval', explode(',', trim($selectedCustomerGroupIds))) : null;
    }

    /**
     * Returns "sort" parameter from the request or $this->getDefaultSorting() if no parameter was given.
     *
     * @return array|null
     */
    protected function getSorting()
    {
        $sorting = $this->Request()->getParam('sort');

        return ($sorting) ? $sorting : $this->getDefaultSorting();
    }

    /**
     * This function must be implement by its SubClass.
     * Returns a default sorting parameter as an array in the following format.
     *
     * array(
     *      array(
     *          'property' => 'someValue',
     *          'direction' => 'DESC',
     *      ),
     *      array(
     *          'property' => 'someOtherValue',
     *          'direction' => 'DESC',
     *      ),
     * )
     *
     * @return array|null
     */
    abstract protected function getDefaultSorting();
}
