<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document;

use BadMethodCallException;

/**
 * Class that represents a page layout for a label page
 */
class GridPaperLayout extends PaperLayout
{
    /**
     * @var float $pageMarginTopInMillimeters
     */
    public $pageMarginTopInMillimeters;

    /**
     * @var float $pageMarginLeftInMillimeters
     */
    public $pageMarginLeftInMillimeters;

    /**
     * @var float $cellHeightInMillimeters
     */
    public $cellHeightInMillimeters;

    /**
     * @var float $cellWidthInMillimeters
     */
    public $cellWidthInMillimeters;

    /**
     * @var int $columnAmount
     */
    public $columnAmount;

    /**
     * @var int $rowAmount
     */
    public $rowAmount;

    /**
     * @var float $horizontalCellSpacingInMillimeters
     */
    public $horizontalCellSpacingInMillimeters;

    /**
     * @var float $verticalCellSpacingInMillimeters
     */
    public $verticalCellSpacingInMillimeters;

    /**
     * Lists all available grid paper layouts
     *
     * @return array
     */
    public static function getDefaultGridPaperLayoutIdentifiers()
    {
        return [
            'avery_4782_2x4',
            'avery_6171_3x6',
            'ebl_060x030_3x7',
            'ebl_048x025_4x10',
            'avery_4736_4x12',
            'avery_4732_5x16',
        ];
    }

    /**
     * Returns all GridPaperLayouts
     *
     * @return GridPaperLayout[]
     */
    public static function getDefaultGridPaperLayouts()
    {
        $identifiers = static::getDefaultGridPaperLayoutIdentifiers();

        return array_map(function ($identifier) {
            return static::createDefaultGridPaperLayout($identifier);
        }, $identifiers);
    }

    /**
     * Returns the GridPaperLayout for $layoutIdentifier
     *
     * @param string $layoutIdentifier
     * @return GridPaperLayout
     * @throws \InvalidArgumentException When the identifier is not a known GridPaperLayout
     */
    public static function createDefaultGridPaperLayout($layoutIdentifier)
    {
        $layout = new self();
        $layout->snippetNamespace = static::SNIPPET_NAMESPACE;

        switch ($layoutIdentifier) {
            case 'avery_4782_2x4':
                $layout->identifier = 'avery_4782_2x4';
                $layout->snippetName = 'paper_layout/avery_4782_2x4/name';
                $layout->pageMarginLeftInMillimeters = 8;
                $layout->pageMarginTopInMillimeters = 12;
                $layout->verticalCellSpacingInMillimeters = 0;
                $layout->horizontalCellSpacingInMillimeters = 0;
                $layout->rowAmount = 4;
                $layout->columnAmount = 2;
                $layout->cellHeightInMillimeters = 67.7;
                $layout->cellWidthInMillimeters = 97;
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                break;
            case 'avery_6171_3x6':
                $layout->identifier = 'avery_6171_3x6';
                $layout->snippetName = 'paper_layout/avery_6171_3x6/name';
                $layout->pageMarginLeftInMillimeters = 9.5;
                $layout->pageMarginTopInMillimeters = 13;
                $layout->verticalCellSpacingInMillimeters = 0;
                $layout->horizontalCellSpacingInMillimeters = 0;
                $layout->rowAmount = 6;
                $layout->columnAmount = 3;
                $layout->cellHeightInMillimeters = 45;
                $layout->cellWidthInMillimeters = 64;
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                break;
            case 'ebl_060x030_3x7':
                $layout->identifier = 'ebl_060x030_3x7';
                $layout->snippetName = 'paper_layout/ebl_060x030_3x7/name';
                $layout->pageMarginLeftInMillimeters = 10;
                $layout->pageMarginTopInMillimeters = 19.5;
                $layout->verticalCellSpacingInMillimeters = 8;
                $layout->horizontalCellSpacingInMillimeters = 5;
                $layout->rowAmount = 7;
                $layout->columnAmount = 3;
                $layout->cellHeightInMillimeters = 30;
                $layout->cellWidthInMillimeters = 60;
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                break;
            case 'ebl_048x025_4x10':
                $layout->identifier = 'ebl_048x025_4x10';
                $layout->snippetName = 'paper_layout/ebl_048x025_4x10/name';
                $layout->pageMarginLeftInMillimeters = 8;
                $layout->pageMarginTopInMillimeters = 23.5;
                $layout->verticalCellSpacingInMillimeters = 0;
                $layout->horizontalCellSpacingInMillimeters = 0;
                $layout->rowAmount = 10;
                $layout->columnAmount = 4;
                $layout->cellHeightInMillimeters = 25.4;
                $layout->cellWidthInMillimeters = 48.5;
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                break;
            case 'avery_4736_4x12':
                $layout->identifier = 'avery_4736_4x12';
                $layout->snippetName = 'paper_layout/avery_4736_4x12/name';
                $layout->pageMarginLeftInMillimeters = 10;
                $layout->pageMarginTopInMillimeters = 21.9;
                $layout->verticalCellSpacingInMillimeters = 0;
                $layout->horizontalCellSpacingInMillimeters = 2.7333;
                $layout->rowAmount = 12;
                $layout->columnAmount = 4;
                $layout->cellHeightInMillimeters = 21.1;
                $layout->cellWidthInMillimeters = 45.7;
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                break;
            case 'avery_4732_5x16':
                $layout->identifier = 'avery_4732_5x16';
                $layout->snippetName = 'paper_layout/avery_4732_5x16/name';
                $layout->pageMarginLeftInMillimeters = 11;
                $layout->pageMarginTopInMillimeters = 12.6;
                $layout->verticalCellSpacingInMillimeters = 0;
                $layout->horizontalCellSpacingInMillimeters = 2.5;
                $layout->rowAmount = 16;
                $layout->columnAmount = 5;
                $layout->cellHeightInMillimeters = 16.9;
                $layout->cellWidthInMillimeters = 35.6;
                $layout->pageOrientation = self::PAGE_ORIENTATION_PORTRAIT;
                $layout->pageFormat = self::PAGE_FORMAT_DIN_A4;
                $layout->pageWidthInMillimeters = 210;
                $layout->pageHeightInMillimeters = 297;
                break;
            default:
                throw new \InvalidArgumentException(sprintf(
                    'The identifier "%s" is not a valid identifier for a GridPaperLayout.',
                    $layoutIdentifier
                ));
        }

        return $layout;
    }

    /**
     * Prevent the usage of parent function
     *
     * @inheritdoc
     * @throws \BadMethodCallException Always
     */
    public static function createDefaultPaperLayout($layoutIdentifier)
    {
        throw new BadMethodCallException('You used the "createDefaultPaperLayout" method on the GridPaperLayout struct. Please use the "createDefaultGridPaperLayout" method instead.');
    }

    /**
     * @return array
     */
    public function toArray()
    {
        return array_merge(
            parent::toArray(),
            [
                'cellHeightInMillimeters' => $this->cellHeightInMillimeters,
                'cellWidthInMillimeters' => $this->cellWidthInMillimeters,
                'columnAmount' => $this->columnAmount,
                'horizontalCellSpacingInMillimeters' => $this->horizontalCellSpacingInMillimeters,
                'pageMarginLeftInMillimeters' => $this->pageMarginLeftInMillimeters,
                'pageMarginTopInMillimeters' => $this->pageMarginTopInMillimeters,
                'rowAmount' => $this->rowAmount,
                'verticalCellSpacingInMillimeters' => $this->verticalCellSpacingInMillimeters,
            ]
        );
    }
}
