<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Classes;

use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

/**
 * This is a helper class for rendering grid layout based documents. That is,
 * documents which have a certain number of rows and columns. It provides methods
 * to render the HTML of the document as well as a mPDF document, which can be
 * displayed in the browser as a PDF file.
 */
class GridDocumentRenderer
{

    /**
     * Renders the HTML of the document using 'renderHTML' and uses that HTML to
     * create a new mPDF files with the given format, which is finally returned.
     * In order to display the mPDF document in the browser, just call
     *
     * $dompdf = \Shopware\Plugins\ViisonShippingCommon\Classes\GridDocumentRenderer::renderPDF(...);
     * $dompdf->stream('filename.pdf');
     *
     * @param string $templatePath The absolute path to the template, that will be rendered.
     * @param string $layout The layout of the label paper, e.g. '4x12', '5x16'
     * @param array $items An array of items, which will be rendered. The content of the array is arbitrary.
     * @param int $offsetX (optional) The horizontal offset of the first cell that should be filled.
     * @param int $offsetY (optional) The vertical offset of the first cell that should be filled.
     * @param string $format (optional) The paper format, which will be used to render the PDF file.
     * @param array $extraTemplateParams An associative array containing extra parameters to be passed to the template.
     * @return \Dompdf\Dompdf The rendered mPDF document.
     */
    public static function renderPDF($templatePath, $layout, array $items, $offsetX = 0, $offsetY = 0, $format = 'A4', array $extraTemplateParams = array())
    {
        // Create a new PDF file using the rendered HTML
        $html = self::renderHTML($templatePath, $layout, $items, $offsetX, $offsetY, $extraTemplateParams);

        $dompdf = ViisonCommonUtil::createDompdfInstance();
        $dompdf->setPaper($format);
        $dompdf->loadHtml($html);
        $dompdf->render();

        return $dompdf;
    }

    /**
     * Loads the template from the given path and assigns both the offset and the items
     * to it, before it renders the HTML and returns it.
     *
     * @param string $templatePath The absolute path to the template, that will be rendered.
     * @param string $layout The layout of the label paper, e.g. '4x12', '5x16'
     * @param array $items An array of items, which will be rendered. The content of the array is arbitrary.
     * @param int $offsetX (optional) The horizontal offset of the first cell that should be filled.
     * @param int $offsetY (optional) The vertical offset of the first cell that should be filled.
     * @param array $extraTemplateParams An associative array containing extra parameters to be passed to the template.
     * @return string The HTML of the rendered document.
     */
    public static function renderHTML($templatePath, $layout, array $items, $offsetX = 0, $offsetY = 0, array $extraTemplateParams = array())
    {
        // Render the document to HTML
        $templateEngine = clone Shopware()->Template();
        // The template differs depending on the layout, so make sure that different caches are used per layout
        $templateEngine->compile_id = $layout;
        $template = $templateEngine->createTemplate($templatePath);
        $template->assign('offsetX', $offsetX);
        $template->assign('offsetY', $offsetY);
        $template->assign('items', $items);
        $template->assign('layout', $layout);
        $template->addTemplateDir(__DIR__ . '/../Documents');
        $template->addTemplateDir(dirname($templatePath));
        foreach ($extraTemplateParams as $key => $value) {
            $template->assign($key, $value);
        }
        $html = $template->fetch();

        return $html;
    }
}
