<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Classes;

use Shopware\CustomModels\ViisonShippingCommon\Shipment\Document;
use Shopware\CustomModels\ViisonShippingCommon\Shipment\Shipment;

/**
 * This is an implementation of the ShippingLabelCreationResult interface defined in ViisonPickwareMobile
 * (and the legacy interface in ViisonPickwareConnector). It extends the basic interface by a new method
 * to add one document at a time.
 */
class ShippingLabelCreationResult implements
    \ViisonPickwareConnector_API_ShippingLabelCreationResult,
    \ViisonPickwareMobile_Interfaces_ShippingProvider_ShippingLabelCreationResult
{

    /**
     * @var null|string
     */
    private $newTrackingCode = null;

    /**
     * @var null|string
     */
    private $trackingCodes = null;

    /**
     * @var null|string
     */
    private $shippingLabelId = null;

    /**
     * @var bool
     */
    private $isCashOnDeliveryShipment = false;

    /**
     * @var array
     */
    private $documents = array();

    /**
     * Static constructor used to create ShippingLabelCreationResult from Shipment.
     * ( Bridge between Pickware and the Adapter ).
     *
     * @param Shipment $shipment
     * @return ShippingLabelCreationResult
     * @throws \Exception
     */
    public static function initializeFromShipment(Shipment $shipment)
    {
        $trackingCode = $shipment->getTrackingCode();
        $order = $shipment->getOrder();

        /**
         * Get the tracking codes from the order table.
         */
        if (!empty($order)) {
            // Get existing tracking codes
            $trackingCodes = $order->getTrackingCode();
        } else {
            $trackingCodes = $trackingCode;
        }

        $result = new ShippingLabelCreationResult($shipment->getTrackingCode(), $trackingCodes);

        $documents = $shipment->getDocuments();
        /** @var Document $document */
        foreach ($documents as $document) {
            $result->addDocument(
                new ShippingDocument(
                    $document->getType(),
                    $trackingCode,
                    $document->getPageSize(),
                    $document->getDocumentTypeId(),
                    $document->getGuid()
                )
            );
        }

        return $result;
    }

    /**
     * @param string $newTrackingCode The new tracking code, which was generated during the creation process.
     * @param string $trackingCodes All comma separated tracking codes.
     * @param string|null $shippingLabelId Optional parameter if the Product does not support Tracking Code, in that case use getShippingLabelId.
     */
    public function __construct($newTrackingCode, $trackingCodes, $shippingLabelId = null)
    {
        $this->newTrackingCode = $newTrackingCode;
        $this->trackingCodes = $trackingCodes;
        $this->shippingLabelId = $shippingLabelId;
    }

    /* Override */
    public function getNewTrackingCode()
    {
        return $this->newTrackingCode;
    }

    /* Override */
    public function getTrackingCodes()
    {
        return $this->trackingCodes;
    }

    /* Override */
    public function getDocuments()
    {
        return $this->documents;
    }

    /**
     * @return string
     */
    public function getShippingLabelId()
    {
        return $this->shippingLabelId;
    }

    /**
     * @return bool
     */
    public function isCashOnDeliveryShipment()
    {
        return $this->isCashOnDeliveryShipment;
    }

    /**
     * @param bool $isCashOnDeliveryShipment
     */
    public function setIsCashOnDeliveryShipment($isCashOnDeliveryShipment)
    {
        $this->isCashOnDeliveryShipment = $isCashOnDeliveryShipment;
    }

    /**
     * Adds a new document to this result.
     *
     * @param \ViisonPickwareConnector_API_ShippingDocument $document The document, which shall be added to this result.
     */
    public function addDocument(\ViisonPickwareConnector_API_ShippingDocument $document)
    {
        $this->documents[] = $document;
    }
}
