<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Components\Export;

use DateTime;

class Information
{
    const DEFAULT_CURRENCY = 'EUR';

    /**
     * @var float
     */
    protected $netInvoiceAmount = 0.0;

    /**
     * @var string
     */
    protected $currency = self::DEFAULT_CURRENCY;

    /**
     * @var DateTime
     */
    protected $date;

    /**
     * @var string
     */
    protected $description = '';

    /**
     * @var string
     */
    protected $invoiceNumber = '';

    /**
     * @var string
     */
    protected $orderNumber = '';

    /**
     * @var float
     */
    protected $invoiceShippingAmount = 0.0;

    /**
     * @var Position[] $positions
     */
    protected $positions = [];

    /**
     * @var string
     */
    protected $vatNumber = '';

    public function __construct()
    {
        $this->date = new DateTime();
    }

    /**
     * @return float
     */
    public function getNetInvoiceAmount()
    {
        return $this->netInvoiceAmount;
    }

    /**
     * @param float $netInvoiceAmount
     * @return Information
     */
    public function setNetInvoiceAmount($netInvoiceAmount)
    {
        $this->netInvoiceAmount = $netInvoiceAmount;

        return $this;
    }

    /**
     * @return string
     */
    public function getCurrency()
    {
        return $this->currency;
    }

    /**
     * @param string $currency
     * @return Information
     */
    public function setCurrency($currency)
    {
        $this->currency = $currency ?: self::DEFAULT_CURRENCY;

        return $this;
    }

    /**
     * @return DateTime
     */
    public function getDate()
    {
        return $this->date;
    }

    /**
     * @param DateTime $date
     * @return Information
     */
    public function setDate($date)
    {
        $this->date = $date;

        return $this;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @param string $description
     * @return Information
     */
    public function setDescription($description)
    {
        $this->description = $description;

        return $this;
    }

    /**
     * @return string
     */
    public function getInvoiceNumber()
    {
        return $this->invoiceNumber;
    }

    /**
     * @param string $invoiceNumber
     * @return Information
     */
    public function setInvoiceNumber($invoiceNumber)
    {
        $this->invoiceNumber = $invoiceNumber;

        return $this;
    }

    /**
     * @return string
     */
    public function getOrderNumber()
    {
        return $this->orderNumber;
    }

    /**
     * @param string $orderNumber
     * @return Information
     */
    public function setOrderNumber($orderNumber)
    {
        $this->orderNumber = $orderNumber;

        return $this;
    }

    /**
     * @return float
     */
    public function getInvoiceShippingAmount()
    {
        return $this->invoiceShippingAmount;
    }

    /**
     * @param float $invoiceShippingAmount
     * @return Information
     */
    public function setInvoiceShippingAmount($invoiceShippingAmount)
    {
        $this->invoiceShippingAmount = $invoiceShippingAmount;

        return $this;
    }

    /**
     * @return Position[]
     */
    public function getPositions()
    {
        return $this->positions;
    }

    /**
     * @param Position[] $positions
     * @return Information
     */
    public function setPositions($positions)
    {
        $this->positions = $positions;

        return $this;
    }

    /**
     * @param Position $position
     * @return Information
     */
    public function addPosition(Position $position)
    {
        $this->positions[] = $position;

        return $this;
    }

    /**
     * @return string
     */
    public function getVatNumber()
    {
        return $this->vatNumber;
    }

    /**
     * @param string $vatNumber
     * @return Information
     */
    public function setVatNumber($vatNumber)
    {
        $this->vatNumber = $vatNumber;

        return $this;
    }

    /**
     * @return float
     */
    public function getAccumulatedPrice()
    {
        return array_sum(array_map(function ($position) {
            /** @var Position $position */
            return $position->getTotalPrice();
        }, $this->positions));
    }

    /**
     * @return int
     */
    public function getTotalNumberOfItems()
    {
        return array_sum(array_map(function ($position) {
            /** @var Position $position */
            return $position->getQuantity();
        }, $this->positions));
    }
}
