// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * The main container of the dispatch service provider tab containing a button to open the LabelConfirmWindow
 * and a list of all existing shipping labels of the active record(order).
 */
//{namespace name="backend/viison_shipping_common_order/order"}
Ext.define('Shopware.apps.ViisonShippingCommonOrder.view.detail.Tab', {

    /**
     * Extend from the standard ExtJS 4.
     */
    extend: 'Ext.container.Container',

    /**
     * List of short aliases for class names. Most useful for defining xtypes for widgets.
     */
    alias: 'widget.order-viison-shipping-common-panel',

    /**
     * An optional extra CSS class that will be added to this component's Element.
     */
    cls: Ext.baseCSSPrefix + 'order-viison-shipping-common-panel',

    /**
     * Defines the component layout.
     */
    layout: 'auto',

    /**
     * Add padding.
     */
    padding: 10,

    /**
     * Enable auto scrolling.
     */
    autoScroll: true,

    /**
     * Supports return labels.
     */
    doesSupportReturnLabel: true,

    /**
     * The grid panel used for creation
     */
    gridView: 'Shopware.apps.ViisonShippingCommonOrder.view.list.Label',

    /**
     * Tells if adapter supports bottom toolbar.
     */
    adapterSupportsBottomToolbar: false,

    /**
     * Contains all snippets for the view component.
     */
    snippets: {
        gridTitle: '{s name="tab/grid_title"}{/s}',
        addLabelButton: {
            title: '{s name="tab/add_label_button/title"}{/s}',
            enabledTooltip: '{s name="tab/add_label_button/enabled_tooltip"}{/s}',
            disabledTooltip: '{s name="tab/add_label_button/disabled_tooltip"}{/s}',
        },
        addReturnLabelButton: {
            title: '{s name="tab/add_return_label_button/title"}{/s}',
            enabledTooltip: '{s name="tab/add_return_label_button/enabled_tooltip"}{/s}',
            disabledTooltip: '{s name="tab/add_return_label_button/disabled_tooltip"}{/s}',
        },
        mergeLabelsFromOrder: {
            title: '{s name="tab/merge_labels_from_order/title"}{/s}',
            enabledTooltip: '{s name="tab/merge_labels_from_order/enabled_tooltip"}{/s}',
            disabledTooltip: '{s name="tab/merge_labels_from_order/disabled_tooltip"}{/s}',
            error: {
                title: '{s name="tab/merge_labels_from_order/error/title"}{/s}',
                message: '{s name="tab/merge_labels_from_order/error/message"}{/s}',
            },
        },
        loadMaskMessage: '{s name="tab/loadMask"}{/s}',
    },

    /**
     * The main initializer registering events and adding view components.
     */
    initComponent: function () {
        // Create a hidden load mask
        this.loadMask = new Ext.LoadMask(this, {
            msg: this.snippets.loadMaskMessage,
        });
        this.loadMask.hide();

        // Initialize a new shipping label store and load all labels for the current order
        this.shippingLabelStore = Ext.create(this.shippingLabelStoreClass);

        this.shippingLabelStore.getProxy().extraParams = {
            orderId: this.initialConfig.record.getId(),
        };

        this.shippingLabelStore.load({
            scope: this,
            callback: function () {
                this.shippingLabelStore.isLoaded = true;
            },
        });

        this.items = [
            this.createToolbar(),
            this.createLabelGrid(),
        ];

        this.items = this.items.concat(
            this.createAdditionalItems()
        );

        this.callParent(arguments);
    },

    /**
     * Creates a toolbar containing a button to add labels.
     *
     * @return The created toolbar.
     */
    createToolbar: function () {
        var me = this;

        var items = [];

        // Add label button
        me.addLabelButton = Ext.create('Ext.button.Button', {
            iconCls: 'sprite-plus-circle-frame',
            text: me.snippets.addLabelButton.title,
            action: 'showLabelConfirmWindow',
            tooltip: me.snippets.addLabelButton.enabledTooltip,
            enabledTooltip: me.snippets.addLabelButton.enabledTooltip,
            disabledTooltip: me.snippets.addLabelButton.disabledTooltip,
            handler: function () {
                me.fireEvent('showLabelConfirmWindow', me, me.record, me.shippingLabelStore);
            },
        });

        items.push(me.addLabelButton);

        if (me.doesSupportReturnLabel) {
            // Add return label button
            me.addReturnLabelButton = Ext.create('Ext.button.Button', {
                iconCls: 'sprite-arrow-return-180',
                text: me.snippets.addReturnLabelButton.title,
                action: 'addReturnLabel',
                tooltip: me.snippets.addReturnLabelButton.enabledTooltip,
                enabledTooltip: me.snippets.addReturnLabelButton.enabledTooltip,
                disabledTooltip: me.snippets.addReturnLabelButton.disabledTooltip,
                handler: function () {
                    me.fireEvent('showReturnLabelConfirmWindow', me, me.record, me.shippingLabelStore);
                },
            });
            items.push({ xtype: 'tbfill' }, me.addReturnLabelButton);
        }

        // Create toolbar and add buttons
        return Ext.create('Ext.toolbar.Toolbar', {
            dock: 'top',
            ui: 'shopware-ui',
            margin: '0 0 10px 0',
            style: {
                padding: 0,
                backgroundColor: 'transparent',
            },
            items: items,
        });
    },

    /**
     * Creates the label grid displaying all generated labels.
     *
     * @return The created grid.
     */
    createLabelGrid: function () {
        var me = this;

        return Ext.create(me.gridView || 'Shopware.apps.ViisonShippingCommonOrder.view.list.Label', {
            store: me.shippingLabelStore,
            record: me.record,
            dispatchServiceProviderPanel: me,
            showCustomerAddress: false,
            minHeight: 250,
            minWidth: 250,
            region: 'center',
            title: me.snippets.gridTitle,
            style: {
                'margin-bottom': '10px',
            },
        });
    },

    /**
     * Additional items to create for the tab panel. Easily overridden by every adapter.
     *
     * @return array
     */
    createAdditionalItems: function () {
        var items = [];

        if (this.adapterSupportsBottomToolbar) {
            items.push(this.createBottomToolbar());
        }

        return items;
    },

    /**
     * Creates bottom toolbar.
     *
     * @return Ext.toolbar.Toolbar
     */
    createBottomToolbar: function () {
        return Ext.create('Ext.toolbar.Toolbar', {
            dock: 'top',
            ui: 'shopware-ui',
            margin: '0 0 10px 0',
            style: {
                padding: 0,
                backgroundColor: 'transparent',
            },
            items: this.getItemsForBottomToolbar(),
        });
    },

    /**
     * @return array
     */
    getItemsForBottomToolbar: function () {
        return [
            this.createMergedLabelsFromOrderButton(
                this.snippets.mergeLabelsFromOrder.error,
                this.snippets.mergeLabelsFromOrder.disabledTooltip,
                this.snippets.mergeLabelsFromOrder.title,
                this.snippets.mergeLabelsFromOrder.enabledTooltip
            ),
        ];
    },

    /**
     * @return Ext.button.Button
     */
    createMergedLabelsFromOrderButton: function (error, disabledTooltip, title, enabledTooltip) {
        var buttonHandler = function () {
            this.fireEvent(
                'getAllLabelsFromOrderAsPdf',
                this.record.getId(),
                this.shippingLabelStore,
                error.title,
                error.message
            );
        };

        return Ext.create('Ext.button.Button', {
            iconCls: 'sprite-inbox-download',
            text: title,
            action: 'getAllLabelsFromOrderAsPdf',
            tooltip: enabledTooltip,
            enabledTooltip: enabledTooltip,
            disabledTooltip: disabledTooltip,
            handler: buttonHandler.bind(this),
        });
    },

});
