<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonDHL\Classes;

use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\FormattedTranslatableThrowable;

class DHLException extends \Exception implements FormattedTranslatableThrowable
{
    /**
     * @var string
     */
    private $snippetName;

    /**
     * Stores additional error messages whose origin is the webservice
     *
     * @var array
     */
    private $additionalInformation;

    /**
     * @param string $message
     * @param int $code
     * @param string $snippetName
     * @param array $additionalInformation
     */
    public function __construct($message, $code, $snippetName, array $additionalInformation = [])
    {
        $this->snippetName = $snippetName;
        $this->additionalInformation = $additionalInformation;

        parent::__construct($message, $code);
    }

    /**
     * @return string
     */
    public function getSnippetNamespace()
    {
        return 'exceptions/viison_dhl';
    }

    /**
     * @return string
     */
    public function getSnippetName()
    {
        return $this->snippetName;
    }

    /**
     * @param $translatedMessage
     * @return string
     */
    public function format($translatedMessage)
    {
        if (!empty($this->additionalInformation)) {
            $translatedMessage = vsprintf($translatedMessage, $this->additionalInformation);
        }

        return $translatedMessage;
    }

    /**
     * @return DHLException
     */
    public static function fileWriteFailed()
    {
        return new self(
            'The saving of the label failed, the directory permissions are likely wrong.',
            3,
            'fileWriteFailed'
        );
    }

    /**
     * @return DHLException
     */
    public static function currencyFactorNotConfigured($currency)
    {
        return new self(
            'The currency factor is not configured for ' . $currency . '.',
            4,
            'currencyFactorNotConfigured',
            [
                $currency,
            ]
        );
    }

    /**
     * @return DHLException
     */
    public static function gkpCredentialsMissing()
    {
        return new self(
            'Credentials for the DHL business customer portal not found. Please check the configuration of the DHL adapter.',
            5,
            'gkpCredentialsMissing'
        );
    }

    /**
     * @return DHLException
     */
    public static function preferredNeighbourAndLocationSetTogether()
    {
        return new self(
            'Preferred location and preferred neighbour can not be set together.',
            6,
            'preferredNeighborAndLocationSet'
        );
    }
}
