<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonDHL\Classes\DHLPortalConfigScraper;

use Shopware\Commands\ShopwareCommand;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * A CLI command that verifies a DHL account and returns some customer information.
 */
class GetDHLAccountInformationCommand extends ShopwareCommand
{
    /**
     * Setup the command name, arguments and help text.
     */
    public function configure()
    {
        $this
            ->setName('viison:dhl:accountinfo')
            ->setDescription('Get DHL GK account info.')
            ->addArgument('user', InputArgument::REQUIRED, 'DHL Geschäftskundenportal user name')
            ->addArgument('password', InputArgument::REQUIRED, 'DHL Geschäftskundenportal password')
            ->setHelp(<<<EOF
<info>%command.name%</info> uses the supplied credentials to log into the DHL Geschäftskundenportal and extract the
customer number and the billing numbers from the portal.
EOF
            );
    }

    /**
     * Run the DHL scraping.
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $user = $input->getArgument('user');
        $password = $input->getArgument('password');
        $debug = $output->getVerbosity() === OutputInterface::VERBOSITY_DEBUG;
        $dhlService = new DhlConfigScraper($user, $password);
        $output->writeln('<info>Logging into the DHL Geschäftskundenportal...</info>');

        /** @var DhlContractData $dhlContactData */
        $dhlContractData = $dhlService->scrape();
        $output->writeln(sprintf('<info>Extracted customer number:</info> %s', $dhlContractData->getCustomerNumber()));
        $billingNumbers = $dhlContractData->getBillingNumbers();
        if (empty($billingNumbers)) {
            $output->writeln('<error>No billing numbers found.</error>');
        } else {
            $output->writeln('<info>Billing numbers:</info>');
            foreach ($billingNumbers as $identifier => $billingNumber) {
                $output->writeln(sprintf('  %s: %s', $identifier, $billingNumber));
            }
        }
    }
}
