<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonDHL\Classes\DhlRetoureApi;

use Exception;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\LocalizableThrowable;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\MessageLocalization;

class DhlRetoureException extends Exception implements LocalizableThrowable
{
    use MessageLocalization;

    /**
     * @param string $message
     * @param string $snippetName
     * @param array $snippetArguments
     */
    public function __construct($message, $snippetName, array $snippetArguments = [])
    {
        $this->snippetName = $snippetName;
        $this->snippetNamespace  = 'exceptions/viison_dhl_retoure';
        $this->snippetArguments = $snippetArguments;

        parent::__construct(vsprintf($message, $snippetArguments));
    }

    /**
     * @return self
     */
    public static function webserviceOffline()
    {
        return new self(
            'The DHL Retoure web service is currently unavailable. Please try again later.',
            'webserviceOffline'
        );
    }

    /**
     * @return self
     */
    public static function webserviceUnauthorized()
    {
        return new self(
            'You do not currently have the necessary authorisations to use the "Retoure" function of the DHL ' .
            'business customer portal. Please contact your colleagues at DHL by phone ' .
            'to activate the function: 0228 - 76 36 76 79 (selection 2)',
            'webserviceUnauthorized'
        );
    }

    /**
     * @param DhlRetoureApiResponse $response
     * @return self
     */
    public static function unexpectedResponse(DhlRetoureApiResponse $response)
    {
        return new self(
            'The DHL Retoure web service returned an unexpected response which could not be parsed. Further information:<br />%s',
            'unexpectedResponse',
            [$response->getRawResponse()]
        );
    }

    /**
     * @param string $error
     * @return self
     */
    public static function webserviceRequestFailed($error)
    {
        return new self(
            'The Return label creation failed. The following error has occurred:<br />%s',
            'webserviceRequestFailed',
            [$error]
        );
    }

    /**
     * @return self
     */
    public static function invalidReceiver()
    {
        return new self(
            'The Return label creation failed. An invalid return recipient is configured.',
            'invalidReceiver'
        );
    }

    /**
     * @return self
     */
    public static function freeFormLabelNotSupportedForSwissCustomers()
    {
        return new self(
            'Unfortunately, the creation of free form return labels for customers from Switzerland is currently not supported.',
            'freeFormLabelNotSupportedForSwissCustomers'
        );
    }

    /**
     * @param string $countryName
     * @return self
     */
    public static function unsupportedCountry($countryName)
    {
        return new self(
            'Für das Land %s werden Rücksendungen derzeit leider nicht unterstützt.',
            'unsupportedCountry',
            [$countryName]
        );
    }

    /**
     * @param string $countryName
     * @return self
     */
    public static function receiverIdForCountryNotConfigured($countryName)
    {
        return new self(
            'Der Retourenempfänger für %s ist nicht gesetzt. Bitte prüfen Sie die Konfiguration des DHL Adapters.',
            'receiverIdForCountryNotConfigured',
            [$countryName]
        );
    }
}
