<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonDHL\Classes;

use Shopware\Plugins\ViisonCommon\Classes\Util\Util;
use Shopware\Plugins\ViisonShippingCommon\Classes\PluginInfo as ShippingCommonPluginInfo;

/**
 * Class PluginInfo
 *
 * This class provides plugin specific information like for example database table
 * names. Instances of this class are injected into generalized classes of the
 * ShippingCommon module.
 *
 * @package Shopware\Plugins\ViisonDHL\Classes
 */
class PluginInfo extends ShippingCommonPluginInfo
{
    /**
     * Returns the table name of the plugin specific order table
     * @return string
     */
    public function getOrderTableName()
    {
        return 's_order_viison_dhl';
    }

    /**
     * Returns the table name of the plugin specific config table
     * @return string
     */
    public function getConfigTableName()
    {
        return 's_core_shops_viison_dhl';
    }

    /**
     * Returns the table name of the plugin specific product table
     * @return string
     */
    public function getProductTableName()
    {
        return 's_viison_dhl_products';
    }

    /**
     * Returns the table name of the plugin specific product mapping table
     * @return string
     */
    public function getProductMappingTableName()
    {
        return 's_premium_dispatch_viison_dhl';
    }

    /**
     * Returns the file prefix to be used for all plugin specific documents (labels, export documents etc.)
     * stored in the shopware documents directory
     * @return string
     */
    public function getFilePrefix()
    {
        return 'dhl';
    }

    /**
     * Returns the dispatch method name to be displayed to the user
     * @return string
     */
    public function getPluginDisplayName()
    {
        return 'DHL Versand';
    }

    /**
     * Returns the plugin name also used as the prefix for all classes (e.g. 'ViisonDHL')
     * @return string
     */
    public function getPluginName()
    {
        return 'ViisonDHL';
    }

    /**
     * Returns the return label email template name
     * @return string
     */
    public function getReturnLabelEmailTemplateName()
    {
        return 'ViisonDHLReturnLabelEmail';
    }

    /**
     * Returns the URL that can be used to track shipments. When tracking a shipment, the respective tracking
     * code of the shipment is appended to this URL.
     *
     * @return string
     */
    public function getTrackingURL()
    {
        return $this->getLocalizedTrackingURL('de');
    }

    /**
     * Return a localized URL that can be used to track shipments.
     *
     * @param string $urlLanguage Language which should be used to track a shipment.
     * @return string
     */
    public function getLocalizedTrackingURL($urlLanguage)
    {
        return ($urlLanguage === 'de') ? 'https://www.dhl.de/de/privatkunden/dhl-sendungsverfolgung.html?piececode=' : 'https://www.dhl.de/en/privatkunden/pakete-empfangen/verfolgen.html?piececode=';
    }

    /**
     * Returns the DHL Express URL that can be used to track shipments. When tracking a shipment, the respective tracking
     * code of the shipment is appended to this URL.
     *
     * @param boolean $isInternational Default false. Flag to determine if the shipment sender is not from Germany.
     * @return string
     */
    public function getExpressTrackingURL($isInternational = false)
    {
        return ($isInternational) ? 'https://www.dhl.com/de-en/home/tracking/tracking-express.html?tracking-id=' : 'https://www.dhl.com/de-de/home/tracking/tracking-express.html?tracking-id=';
    }

    /**
     * Returns the delimiter to be used to separate individual tracking numbers in the parameter send to the
     * tracking URL if multiple shipments are tracked at the same time.
     * @return string
     */
    public function getTrackingNumberDelimiter()
    {
        return ',';
    }

    /**
     * Returns the maximum number of labels per order.
     * @return int
     */
    public function getMaxLabels()
    {
        if (Util::assertMinimumShopwareVersion('5.5.4')) {
            return PHP_INT_MAX;
        }

        return 9;
    }

    /**
     * This method should be overwritten and return true in plugins that support the creation of export documents. Such
     * plugins are expected to add the article attributes viison_country_of_origin and viison_customs_tariff_number.
     * @return bool
     */
    public function pluginSupportsExportDocuments()
    {
        return true;
    }

    /**
     * Get participant codes for supported countries for DHL Shipment
     *
     * NOTE: When we get more countries a DB needs to created where we can store that countries,
     *       right now it isn't necessary
     *
     * @return array Contains ISO code and participant codes of supported countries
     */
    public function getDefinedParticipantCodesForCountries()
    {
        return [
            'DE' => [
                'EPN',
                'EPO',
                'BPC',
                'EPI',
                'BPI',
                'WAP',
                'WPI',
            ],
        ];
    }
}
