// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonDHLShipping.controller.MappingPanel', {

    extend: 'Enlight.app.Controller',

    /**
     * Assign methods to events
     */
    init: function () {
        this.callParent(arguments);
        this.control({
            'viison-dhl-shipping-mapping-panel': {
                changeProductComboBox: this.onChangeProductComboBox,
                changeCreateExportDocumentCheckBox: this.onCreateExportDocumentComboboxChange,
                mappingValuesSetInForm: this.onMappingValuesSetInForm,
            },
        });
    },

    /**
     * Update field constraints when the product was changed using the combobox.
     *
     * @param { Shopware.apps.ViisonDHLShipping.view.edit.MappingPanel } mappingPanel The panel to operate on
     * @param { integer } newValue The new value of the changed field.
     */
    onChangeProductComboBox: function (mappingPanel, newValue) {
        var newProduct = mappingPanel.productStore.getById(newValue);
        var productSupportsFrankatur = this.productSupportsFrankatur(newProduct.get('product'));

        this.updateFieldVisibility(mappingPanel.frankaturCombobox, productSupportsFrankatur);
        this.updateFieldVisibility(mappingPanel.saturdayDeliveryForExpress, newProduct.get('code') === 'EXP' && ['N', '1'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.insuredCheckBoxForDHLExpress, newProduct.get('code') === 'EXP');
        this.updateContainerVisiblity(mappingPanel.visualAgeCheck, ['V01PAK', 'V01PRIO'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.personalHandover, ['V01PAK', 'V01PRIO'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.parcelOutletRouting, ['V01PAK', 'V01PRIO', 'V62WP'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.endorsementTypeCombobox, ['V53WPAK'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.identCheckAge, ['V01PAK'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.postalDeliveryDutyPaid, ['V53WPAK'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.closestDroppointDelivery, ['V53WPAK'].includes(newProduct.get('product')));
        this.updateContainerVisiblity(mappingPanel.signedForByRecipient, ['V01PAK'].includes(newProduct.get('product')));

        var hideCreateExportDocumentCheckbox = productSupportsFrankatur || newProduct.get('code') === 'EXP';
        if (hideCreateExportDocumentCheckbox) {
            mappingPanel.createExportDocumentCheckbox.setValue(false);
        }
        // Do not disable the field or else extjs will not save "false" from above.
        mappingPanel.createExportDocumentCheckbox.setVisible(!hideCreateExportDocumentCheckbox);

        var showIncotermCombobox = mappingPanel.createExportDocumentCheckbox.getValue() || newProduct.get('code') === 'EXP';
        this.updateFieldVisibility(mappingPanel.incotermCombobox, showIncotermCombobox);
    },

    /**
     * Enable the incoterm checkbox if export documents are created.
     *
     * @param { Shopware.apps.ViisonDHLShipping.view.edit.MappingPanel } mappingPanel
     * @param { bool } newValue
     */
    onCreateExportDocumentComboboxChange: function (mappingPanel, newValue) {
        this.updateFieldVisibility(mappingPanel.incotermCombobox, newValue);
    },

    /**
     * Update the incoterm combobox visibility based on the product code or the `createExportDocument` checkbox.
     *
     * @param { Shopware.apps.ViisonDHLShipping.view.edit.MappingPanel } mappingPanel
     */
    onMappingValuesSetInForm: function (mappingPanel) {
        var productId = mappingPanel.getForm().getValues().product;
        var index = mappingPanel.productStore.findExact('id', productId);
        var product = mappingPanel.productStore.getAt(index);
        var showIncotermCombobox = mappingPanel.createExportDocumentCheckbox.getValue() || product.get('code') === 'EXP';
        this.updateFieldVisibility(mappingPanel.incotermCombobox, showIncotermCombobox);
    },

    /**
     * @param { string } productIdentifier
     * @returns { boolean }
     */
    productSupportsFrankatur: function (productIdentifier) {
        var productsWithFrankatur = ['V54EPAK'];

        return productsWithFrankatur.some(function (productWithFrankatur) {
            return productIdentifier === productWithFrankatur;
        });
    },

    /**
     * @param { Ext.form.field } field
     * @param { bool } visible
     */
    updateFieldVisibility: function (field, visible) {
        field.setVisible(visible);
        if (!visible) {
            field.setValue('');
        }
    },

    /**
     * @param { Ext.form.Container } container
     * @param { boolean } visible
     */
    updateContainerVisiblity: function (container, visible) {
        container.setVisible(visible);
        // To be sure that the child values won't be saved.
        container.setDisabled(!visible);
    },
});
