<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Structs;

use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class PluginCompatibilityIssue extends Struct
{
    /**
     * Name of the plugin, which exposes the unsolved requirement.
     *
     * @var string
     */
    protected $plugin;

    /**
     * @var Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint
     */
    protected $unsolvedRequirement;

    public function __construct($plugin, PluginCompatibilityConstraint $unsolvedRequirement)
    {
        $this->plugin = $plugin;
        $this->unsolvedRequirement = $unsolvedRequirement;
    }

    /**
     * Set the value of Plugin
     *
     * @param string plugin
     *
     * @return self
     */
    public function setPlugin($plugin)
    {
        $this->plugin = $plugin;

        return $this;
    }

    /**
     * Set the value of Unsolved Requirement
     *
     * @param Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint unsolvedRequirement
     *
     * @return self
     */
    public function setUnsolvedRequirement(Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint $unsolvedRequirement)
    {
        $this->unsolvedRequirement = $unsolvedRequirement;

        return $this;
    }

    /**
     * Get the value of Plugin
     *
     * @return string
     */
    public function getPlugin()
    {
        return $this->plugin;
    }

    /**
     * Get the value of Unsolved Requirement
     *
     * @return Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint
     */
    public function getUnsolvedRequirement()
    {
        return $this->unsolvedRequirement;
    }

    /**
     * Returns the info message, which will be shown to the
     * user in order to inform him about the compatibility
     * issue. If no custom message for the related constraint
     * is set, a default message will be returned.
     * The message as well a link and linktitle are returned as an array.
     *
     * @return array
     */
    public function getGrowlMessageInformation()
    {
        if ($this->unsolvedRequirement->hasCustomMessage()) {
            $message = $this->unsolvedRequirement->getCustomMessage($this->getPlugin());
        } else {
            $message = sprintf(
                $this->getDefaultMessageSnippet(),
                $this->getPluginDisplayNameIfPossible($this->getPlugin()),
                $this->unsolvedRequirement->getMinimumVersion(),
                $this->getPluginDisplayNameIfPossible($this->unsolvedRequirement->getPlugin())
            );
        }

        $linkTitle = $this->unsolvedRequirement->getLinkTitle();
        $link = $this->unsolvedRequirement->getLink();
        if ($link && !$linkTitle) {
            // No title (button caption) has been set to the link. Use default
            $linkTitle = Shopware()->Snippets()->getNamespace('backend/viison_common_compatibility_check/main')->get('check/notification/additionalinformation');
        }

        $mutableGrowlMessageInfo = [
            'message' => $message,
        ];

        if ($link !== null) {
            $mutableGrowlMessageInfo['detailButton'] = [
                'title' => $linkTitle,
                'link' => $link,
            ];
        }

        return $mutableGrowlMessageInfo;
    }

    /**
     * @param $internalPluginName string A technical plugin name
     * @return string An easier-to-understand name for the plugin, or $internalPluginName if one cannot be determined
     */
    private function getPluginDisplayNameIfPossible($internalPluginName)
    {
        $plugin = ViisonCommonUtil::getPlugin($internalPluginName);

        return $plugin ? $plugin->getLabel() : $internalPluginName;
    }

    /**
     * @return array
     */
    public function toArray()
    {
        return [
            'plugin' => $this->getPlugin(),
            'requirement' => $this->getUnsolvedRequirement()->toArray(),
        ];
    }

    /**
     * Returns the snippet of the default message, which will
     * be shown to the user to inform him about the compatibility
     * issue.
     *
     * @return string
     */
    protected function getDefaultMessageSnippet()
    {
        $snippet = Shopware()->Snippets()->getNamespace('backend/viison_common_compatibility_check/main')->get('check/notification/message');

        return preg_replace('/\\[\\d+\\]/', '%s', $snippet);
    }
}
