<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Classes;

/**
 * Implementations of this abstract class provide plugin specific information
 * like for example database table names. Instances of these are injected into
 * other classes of the ShippingCommon module.
 */
abstract class PluginInfo
{
    /**
     * Separator used for the label Identifier
     */
     const LABEL_IDENTIFIER_SEPARATOR = ':';

    /**
     * Parameter used for shipment id
     */
    const SHIPMENT_PARAM_ID = 'shipmentId';

    /**
     * Parameter used for shipment type
     */
    const SHIPMENT_PARAM_TYPE = 'type';

    /**
     * Returns the table name of the plugin specific order table.
     *
     * @return string
     */
    abstract public function getOrderTableName();

    /**
     * Returns the table name of the plugin specific config table.
     *
     * @return string
     */
    abstract public function getConfigTableName();

    /**
     * Returns the table name of the plugin specific product table.
     *
     * @return string
     */
    abstract public function getProductTableName();

    /**
     * Returns the table name of the plugin specific product mapping table.
     *
     * @return string
     */
    abstract public function getProductMappingTableName();

    /**
     * Returns the file prefix to be used for all plugin specific documents (labels, export documents etc.)
     * stored in the shopware documents directory.
     *
     * @return string
     */
    abstract public function getFilePrefix();

    /**
     * Returns the dispatch method name to be displayed to the user.
     *
     * @return string
     */
    abstract public function getPluginDisplayName();

    /**
     * Returns the plugin name also used as the prefix for all classes (e.g. 'ViisonIntraship').
     *
     * @return string
     */
    abstract public function getPluginName();

    /**
     * Returns the return label email template name.
     *
     * @return string
     */
    abstract public function getReturnLabelEmailTemplateName();

    /**
     * Returns the URL that can be used to track shipments. When tracking a shipment, the respective tracking
     * code of the shipment is appended to this URL.
     *
     * @return string
     */
    abstract public function getTrackingURL();

    /**
     * Returns the delimiter to be used to separate individual tracking numbers in the parameter send to the
     * tracking URL if multiple shipments are tracked at the same time.
     *
     * @return string
     */
    abstract public function getTrackingNumberDelimiter();

    /**
     * Returns the maximum number of labels per order.
     *
     * @return int
     */
    abstract public function getMaxLabels();

    /**
     * This method should be overwritten and return true in plugins that support the creation of export documents. Such
     * plugins are expected to add the article attributes viison_country_of_origin and viison_customs_tariff_number.
     *
     * Note: This method is not abstract and has a default return value to maintain backwards compatibility.
     *
     * @return bool
     */
    public function pluginSupportsExportDocuments()
    {
        return false;
    }

    /**
     * Determines whether the plugin supports mroe than one tracking code in a single tracking URL.
     * Defaults to TRUE, but should be overwritten by plugins, which only support a single tracking
     * code per URL.
     *
     * Note: This method is not abstract and has a default return value to maintain backwards compatibility.
     *
     * @return bool
     */
    public function supportsMultipleTrackingCodesPerUrl()
    {
        return true;
    }

    /**
     * Returns additional parameters, that shall be added to a tracking URL for the given $trackingCode.
     *
     * Note: This method is not abstract and has a default return value to maintain backwards compatibility.
     *
     * @param string $trackingCode
     * @return string
     */
    public function additionalParametersForTrackingUrl($trackingCode)
    {
        return '';
    }

    /**
     * Returns the full qualified, namespaced name of the 'Shipment' model.
     *
     * TODO: Make abstract (one day).
     *
     * @return string
     */
    public function getShipmentModelName()
    {
        return null;
    }

    /**
     * Returns the full qualified, namespaced name of the shipment 'Document' model.
     *
     * TODO: Make abstract (one day).
     *
     * @return string
     */
    public function getShipmentDocumentModelName()
    {
        return null;
    }

    /**
     * Returns the full qualified, namespaced name of the 'Configuration' model.
     *
     * TODO: Make abstract (one day).
     *
     * @return string
     */
    public function getConfigurationModelName()
    {
        return null;
    }

    /**
     * Returns the full qualified, namespaced name of the 'Product' model.
     *
     * TODO: Make abstract (one day).
     *
     * @return string
     */
    public function getProductModelName()
    {
        return null;
    }

    /**
     * Returns the full qualified, namespaced name of the product 'DispatchMapping' model.
     *
     * TODO: Make abstract (one day).
     *
     * @return string
     */
    public function getProductDispatchMappingModelName()
    {
        return null;
    }
}
