<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Classes;

use Shopware\Plugins\ViisonShippingCommon\Classes\PluginInfo;

/**
 * This is an implementation of the ShippingDocument interface defined in ViisonPickwareMobile
 * (and the legacy interface in ViisonPickwareConnector). It meets the special requirements of
 * the dispatch service provider documents by dynamically creating the identifier.
 */
class ShippingDocument implements
    \ViisonPickwareConnector_API_ShippingDocument,
    \ViisonPickwareMobile_Interfaces_ShippingProvider_ShippingDocument
{
    /**
     * @var null|string Type of the document, e.g. label, codcard, export documents ...
     */
    private $type;

    private $trackingCode = null;

    private $pageSize;

    private $shippingLabelIdentifier = null;

    private $documentTypeId = null;

    /**
     * @param string $type The type of this document, e.g. 'exportDocument'.
     * @param string|null $trackingCode The tracking code of this document.
     * @param string $pageSize The page size of the real document, which is represented by this instance.
     * @param int|null $documentTypeId
     * @param string|null $shippingLabelIdentifier Optional Identifier used if the Product does not support Tracking Url
     */
    public function __construct($type, $trackingCode, $pageSize, $documentTypeId = null, $shippingLabelIdentifier = null)
    {
        $this->type = $type;
        $this->trackingCode = $trackingCode;
        $this->pageSize = $pageSize;
        $this->shippingLabelIdentifier = $shippingLabelIdentifier;
        $this->documentTypeId = $documentTypeId;
    }

    /* Override */
    public function getType()
    {
        return $this->type;
    }

    /* Override */
    public function getIdentifier()
    {
        // If the Product does not support Tracking URL use shippingLabelIdentifier for label name creation
        return $this->getShippingLabelIdentifier() ?: $this->type . PluginInfo::LABEL_IDENTIFIER_SEPARATOR . $this->trackingCode;
    }

    /* Override */
    public function getPageSize()
    {
        return $this->pageSize;
    }

    /* Override */
    public function getTrackingCode()
    {
        return $this->trackingCode;
    }

    /**
     * @return null|string
     */
    public function getShippingLabelIdentifier()
    {
        return $this->shippingLabelIdentifier;
    }

    /**
     * @param null|string $shippingLabelIdentifier
     */
    public function setShippingLabelIdentifier($shippingLabelIdentifier)
    {
        $this->shippingLabelIdentifier = $shippingLabelIdentifier;
    }

    /**
     * @return int
     */
    public function getDocumentTypeId()
    {
        return $this->documentTypeId;
    }

    /**
     * @param int $documentTypeId
     */
    public function setDocumentTypeId($documentTypeId)
    {
        $this->documentTypeId = $documentTypeId;
    }
}
