<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Components\Logger;

/**
 * Class to log http request/responses
 *
 * You need to implement abstract functions for filtering sensitive data.
 *
 */
abstract class AbstractHttpRequestLogger extends AbstractLogger
{
    /**
     * Logs the last request with level "debug"
     *
     * @param string $url
     * @param array $request
     * @param array $response
     */
    public function logSuccessfulRequest($url, $request, $response)
    {
        if (isset($request['headers'])) {
            foreach ($request['headers'] as $headerKey => &$headerValue) {
                $headerValue = $this->filterSensitiveDataFromRequestHeaderIfNecessary(
                    $headerKey,
                    $headerValue
                );
            }
        }
        $request['body'] = $this->filterSensitiveDataFromRequestBodyIfNecessary($request['body']);

        $this->logger->debug(
            sprintf('HTTP request to %s executed.', $url),
            [
                'url' => $url,
                'request' => $request,
                'response' => $response,
            ]
        );
    }

    /**
     * Logs a failed request with level "error"
     *
     * @param string $url
     * @param string|array $request
     */
    public function logFailedRequest($url, $request)
    {
        if (isset($request['headers'])) {
            foreach ($request['headers'] as $headerKey => &$headerValue) {
                $headerValue = $this->filterSensitiveDataFromRequestHeaderIfNecessary(
                    $headerKey,
                    $headerValue
                );
            }
        }
        $request['body'] = $this->filterSensitiveDataFromRequestBodyIfNecessary($request['body']);

        $this->logger->error(
            sprintf('HTTP request to %s failed.', $url),
            [
                'url' => $url,
                'request' => $request,
            ]
        );
    }

    /**
     * @param string $headerKey
     * @param string $headerValue
     * @return string
     */
    private function filterSensitiveDataFromRequestHeaderIfNecessary($headerKey, $headerValue)
    {
        if ($this->logSensitiveData) {
            return $headerValue;
        }

        return $this->filterSensitiveDataFromRequestHeader($headerKey, $headerValue);
    }

    /**
     * @param string $body
     * @return string
     */
    private function filterSensitiveDataFromRequestBodyIfNecessary($body)
    {
        if ($this->logSensitiveData) {
            return $body;
        }

        return $this->filterSensitiveDataFromRequestBody($body);
    }

    /**
     * Removes sensitive data from the request header value
     *
     * @param string $headerKey
     * @param string $headerValue
     * @return string
     */
    abstract public function filterSensitiveDataFromRequestHeader($headerKey, $headerValue);

    /**
     * Removes sensitive data from the request body
     *
     * @param string $body
     * @return string
     */
    abstract public function filterSensitiveDataFromRequestBody($body);
}
