<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonShippingCommon\Components\Logger;

/**
 * Class to log SOAP request/responses
 *
 * You need to implement abstract functions for filtering sensitive data.
 */
abstract class AbstractSoapLogger extends AbstractLogger
{
    /**
     * Logs a failed SOAP request with level "error"
     *
     * @param \SoapClient $client
     * @param \SoapFault $soapFault
     */
    public function logFailedRequest(\SoapClient $client, \SoapFault $soapFault)
    {
        $message = sprintf(
            'SOAP action failed with code %s: %s',
            $soapFault->faultcode,
            $soapFault->faultstring
        );

        $this->logger->error(
            $message,
            [
                'error' => [
                    'code' => $soapFault->faultcode,
                    'name' => $soapFault->faultname,
                    'detail' => $soapFault->detail,
                    'actor' => $soapFault->faultactor,
                    'header' => $soapFault->headerfault,
                ],
                'request' => $this->getLastSoapRequestAsArray($client),
            ]
        );
    }

    /**
     * Logs the last SOAP request of $client with level "debug"
     *
     * @param \SoapClient $client
     */
    public function logLastRequest(\SoapClient $client)
    {
        $this->logger->debug(
            'SOAP action executed successfully',
            [
                'request' => $this->getLastSoapRequestAsArray($client),
            ]
        );
    }

    /**
     * @param \SoapClient $client
     * @return array
     */
    private function getLastSoapRequestAsArray(\SoapClient $client)
    {
        if ($this->logSensitiveData) {
            return [
                'requestHeader' => $client->__getLastRequestHeaders(),
                'requestBody' => $client->__getLastRequest(),
                'responseHeader' => $client->__getLastResponseHeaders(),
                'responseBody' => $client->__getLastResponse(),
            ];
        }

        return [
            'requestHeader' => $this->filterSensitiveHeaderData($client->__getLastRequestHeaders()),
            'requestBody' => $this->filterSensitiveBodyData($client->__getLastRequest()),
            'responseHeader' => $this->filterSensitiveHeaderData($client->__getLastResponseHeaders()),
            'responseBody' => $this->filterSensitiveBodyData($client->__getLastResponse()),
        ];
    }

    /**
     * Removes sensitive data from the SOAP HTTP headers.
     *
     * This should be login credentials or API keys
     *
     * @param string $header
     * @return string Filtered headers
     */
    abstract public function filterSensitiveHeaderData($header);

    /**
     * Removes sensitive data from the SOAP HTTP body.
     *
     * This should be login credentials or API keys
     *
     * @param string $body
     * @return string Filtered body
     */
    abstract public function filterSensitiveBodyData($body);
}
