<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * @Shopware\noEncryption
 */

namespace Shopware\CustomModels\ViisonShippingCommon\Configuration;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Shop\Shop;

/**
 * @ORM\MappedSuperclass
 *
 * Note: Every adapter needs to specify the ORM\Table and ORM\Entity annotations in their subclasses of this model:
 *
 *       ORM\Entity(repositoryClass="Shopware\CustomModels\ADAPTER_NAME\Configuration\Repository")
 *       ORM\Table(name="s_plugin_ADAPTER_NAME_configurations")
 */
abstract class Configuration extends ModelEntity
{
    /**
     * Override this method to return an array of property names, whose values must be encrypted before being saved.
     *
     * @return string[]
     */
    abstract public function getEncryptedProperties();

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    protected $id;

    /**
     * @var int
     *
     * @ORM\Column(name="shopId", type="integer", nullable=false)
     */
    protected $shopId;

    /**
     * OWNING SIDE
     *
     * @var Shop
     *
     * @ORM\OneToOne(targetEntity="Shopware\Models\Shop\Shop")
     * @ORM\JoinColumn(name="shopId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $shop;

    /**
     * @var float
     *
     * @ORM\Column(name="defaultShipmentWeight", type="decimal", precision=10, scale=3, nullable=false)
     */
    protected $defaultShipmentWeight = 0.1;

    /**
     * @var bool
     *
     * @ORM\Column(name="calculateWeight", type="boolean", nullable=false)
     */
    protected $calculateWeight = false;

    /**
     * @var float
     *
     * @ORM\Column(name="fillerSurcharge", type="decimal", precision=10, scale=3, nullable=false)
     */
    protected $fillerSurcharge = 0.0;

    /**
     * @var float
     *
     * @ORM\Column(name="packagingSurcharge", type="decimal", precision=10, scale=3, nullable=false)
     */
    protected $packagingSurcharge = 0.0;

    /**
     * @deprecated Since SW 5.5 no encryption is used anymore. Because of backwards compatibility we have to keep this
     * property.
     * @var string
     *
     * @ORM\Column(name="encryptionMethod", type="string", nullable=false)
     */
    protected $encryptionMethod;

    public function __construct()
    {
        $this->encryptionMethod = (extension_loaded('openssl')) ? 'aes-256-cbc' : 'rijndael256';
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return Shop
     */
    public function getShop()
    {
        return $this->shop;
    }

    /**
     * @param Shop $shop
     */
    public function setShop(Shop $shop)
    {
        $this->shop = $shop;
    }

    /**
     * @return float
     */
    public function getDefaultShipmentWeight()
    {
        return $this->defaultShipmentWeight;
    }

    /**
     * @param float $defaultShipmentWeight
     */
    public function setDefaultShipmentWeight($defaultShipmentWeight)
    {
        $this->defaultShipmentWeight = $defaultShipmentWeight;
    }

    /**
     * @return bool
     */
    public function isCalculateWeight()
    {
        return $this->calculateWeight;
    }

    /**
     * @param bool $calculateWeight
     */
    public function setCalculateWeight($calculateWeight)
    {
        $this->calculateWeight = $calculateWeight;
    }

    /**
     * @return float
     */
    public function getFillerSurcharge()
    {
        return $this->fillerSurcharge;
    }

    /**
     * @param float $fillerSurcharge
     */
    public function setFillerSurcharge($fillerSurcharge)
    {
        $this->fillerSurcharge = $fillerSurcharge;
    }

    /**
     * @return float
     */
    public function getPackagingSurcharge()
    {
        return $this->packagingSurcharge;
    }

    /**
     * @param float $packagingSurcharge
     */
    public function setPackagingSurcharge($packagingSurcharge)
    {
        $this->packagingSurcharge = $packagingSurcharge;
    }

    /**
     * @return string
     */
    public function getEncryptionMethod()
    {
        return $this->encryptionMethod;
    }

    /**
     * @param string $encryptionMethod
     */
    public function setEncryptionMethod($encryptionMethod)
    {
        $this->encryptionMethod = $encryptionMethod;
    }
}
