<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * @Shopware\noEncryption
 */

namespace Shopware\CustomModels\ViisonShippingCommon\Product;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Dispatch\Dispatch as DispatchMethod;
use Shopware\Plugins\ViisonCommon\Classes\Util\Doctrine as DoctrineUtil;
use \InvalidArgumentException;

/**
 * @ORM\MappedSuperclass
 *
 * Note: Every adapter needs to specify the ORM\Table and ORM\Entity annotations in their subclasses of this model:
 *
 *       ORM\Entity(repositoryClass="Shopware\CustomModels\ADAPTER_NAME\Product\Repository")
 *       ORM\Table(name="s_plugin_ADAPTER_NAME_product_dipatch_mappings")
 */
abstract class DispatchMapping extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    protected $id;

    /**
     * @var int
     *
     * @ORM\Column(name="dispatchId", type="integer", nullable=false)
     */
    protected $dispatchId;

    /**
     * OWNING SIDE
     *
     * @var DispatchMethod
     *
     * @ORM\OneToOne(targetEntity="Shopware\Models\Dispatch\Dispatch")
     * @ORM\JoinColumn(name="dispatchId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $dispatch;

    /**
     * @var int
     *
     * @ORM\Column(name="productId", type="integer", nullable=true)
     */
    protected $productId;

    /**
     * OWNING SIDE
     *
     * @var Product
     *
     * Note: This definition has no annotations, because they have to be defined in the subclasses. Re-implement this
     *       property again in your subclass with the following annotation (replace ADAPTER_NAME):
     *
     *       ORM\ManyToOne(targetEntity="Shopware\CustomModels\ADAPTER_NAME\Product\Product", inversedBy="dispatchMappings")
     *       ORM\JoinColumn(name="productId", referencedColumnName="id", onDelete="SET NULL")
     */
    protected $product;

    /**
     * @var bool
     *
     * @ORM\Column(name="exportDocumentRequired", type="boolean", nullable=false)
     */
    protected $exportDocumentRequired = false;

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return DispatchMethod
     */
    public function getDispatch()
    {
        return $this->dispatch;
    }

    /**
     * @param DispatchMethod $dispatch
     */
    public function setDispatch(DispatchMethod $dispatch)
    {
        $this->dispatch = $dispatch;
    }

    /**
     * @return Product
     */
    public function getProduct()
    {
        return $this->product;
    }

    /**
     * @param Product|null $product
     * @throws InvalidArgumentException
     */
    public function setProduct(Product $product = null)
    {
        // Make sure the passed product is part of the same namespace as this dispatch mapping
        if ($product && !DoctrineUtil::customModelsShareNamespace($product, $this)) {
            throw new InvalidArgumentException(
                sprintf('The namespace of the provided $product (%s) must match the namespace of $this dispatch mapping (%s).', get_class($product), get_class($this))
            );
        }

        $this->product = $product;
    }

    /**
     * @return bool
     */
    public function getExportDocumentRequired()
    {
        return $this->exportDocumentRequired;
    }

    /**
     * @param bool $exportDocumentRequired
     */
    public function setExportDocumentRequired($exportDocumentRequired)
    {
        $this->exportDocumentRequired = $exportDocumentRequired;
    }
}
