<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * @Shopware\noEncryption
 */

namespace Shopware\CustomModels\ViisonShippingCommon\Product;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Plugins\ViisonCommon\Classes\Util\Doctrine as DoctrineUtil;
use \InvalidArgumentException;

/**
 * @ORM\MappedSuperclass
 *
 * Note: Every adapter needs to specify the ORM\Table and ORM\Entity annotations in their subclasses of this model:
 *
 *       ORM\Entity(repositoryClass="Shopware\CustomModels\ADAPTER_NAME\Product\Repository")
 *       ORM\Table(name="s_plugin_ADAPTER_NAME_products")
 */
abstract class Product extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    protected $id;

    /**
     * @var string
     *
     * @ORM\Column(name="name", type="string", nullable=false)
     */
    protected $name;

    /**
     * @var string
     *
     * @ORM\Column(name="code", type="string", nullable=false)
     */
    protected $code;

    /**
     * @var bool
     *
     * @ORM\Column(name="packageDimensionsRequired", type="boolean", nullable=false)
     */
    protected $packageDimensionsRequired = false;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * Note: This definition has no annotations, because they have to be defined in the subclasses. Re-implement this
     *       property again in your subclass with the following annotation (replace ADAPTER_NAME):
     *
     *       ORM\OneToMany(targetEntity="Shopware\CustomModels\ADAPTER_NAME\Product\DispatchMapping", mappedBy="product")
     */
    protected $dispatchMappings;

    /**
     */
    public function __construct()
    {
        $this->dispatchMappings = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * @param string $code
     */
    public function setCode($code)
    {
        $this->code = $code;
    }

    /**
     * @return bool
     */
    public function isPackageDimensionsRequired()
    {
        return $this->packageDimensionsRequired;
    }

    /**
     * @param bool $packageDimensionsRequired
     */
    public function setPackageDimensionsRequired($packageDimensionsRequired)
    {
        $this->packageDimensionsRequired = $packageDimensionsRequired;
    }

    /**
     * @return ArrayCollection
     */
    public function getDispatchMappings()
    {
        return $this->dispatchMappings;
    }

    /**
     * @param DispatchMapping $dispatchMapping
     * @throws InvalidArgumentException
     */
    public function addDispatchMapping(DispatchMapping $dispatchMapping)
    {
        if ($this->dispatchMappings->contains($dispatchMapping)) {
            return;
        }

        // Make sure the passed dispatch mapping is part of the same namespace as this product
        if (!DoctrineUtil::customModelsShareNamespace($dispatchMapping, $this)) {
            throw new InvalidArgumentException(
                sprintf('The namespace of the provided $dispatchMapping (%s) must match the namespace of $this product (%s).', get_class($dispatchMapping), get_class($this))
            );
        }

        $this->dispatchMappings->add($dispatchMapping);
        $dispatchMapping->setProduct($this);
    }
}
