// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * The list containing all batch label status objects of the active batch processing.
 * It consists of three columns:
 *    - 'orderNumber', the number of the processed order
 *    - 'trackingCode', showing the tracking code/shipping number of the label
 *    - 'status', providing more detailed information about the status of each label creation process
 */
//{namespace name="backend/viison_shipping_common_order/order"}
Ext.define('Shopware.apps.ViisonShippingCommonOrder.view.BatchLabel.List', {

    /**
     * Extend from the standard ExtJS 4.
     */
    extend: 'Ext.grid.Panel',

    /**
     * List of short aliases for class names. Most useful for defining xtypes for widgets.
     */
    alias: 'widget.order-viison-shipping-common-batch-label-list',

    /**
     * An optional extra CSS class that will be added to this component's Element.
     */
    cls: Ext.baseCSSPrefix + 'order-viison-shipping-common-batch-label-list',

    /**
     * Remove the 'dirty record' red corned from all rows.
     */
    viewConfig: {
        markDirty: false
    },

    /**
     * Contains all snippets for the view component.
     */
    snippets: {
        columns: {
            orderNumber: '{s name="batch_label/list/columns/order_number"}{/s}',
            trackingCode: '{s name="batch_label/list/columns/tracking_code"}{/s}',
            success: '{s name="batch_label/list/columns/success"}{/s}'
        },
        rows: {
            successMessage: '{s name="batch_label/list/rows/success_message"}{/s}',
            openOrderTooltip: '{s name="batch_label/list/rows/open_order_tooltip"}{/s}',
            isNotTrackable: '{s name="label/rows/is_not_trackable"}{/s}'
        }
    },

    /**
     * The icon classes of the batch label success states.
     */
    statusIcons: {
        0: 'c-sprite-tick',
        1: 'c-sprite-exclamation-red'
    },

    /**
     * Icon for warning message that is shown in the status column.
     */
    warningIcon: 'sprite-box--exclamation',

    /**
     * The main initializer setting up this grid by adding the column definitions.
     */
    initComponent: function() {
        this.columns = this.getColumns();

        this.callParent(arguments);
    },

    /**
     * Creates the grid columns.
     *
     * @return The column structure.
     */
    getColumns: function() {
        var me = this;

        return [
            {
                header: me.snippets.columns.orderNumber,
                dataIndex: 'orderNumber',
                width: 90
            }, {
                header: me.snippets.columns.trackingCode,
                dataIndex: 'trackingCode',
                width: 170,
                renderer: me.trackingCodeRenderer
            }, {
                header: me.snippets.columns.success,
                dataIndex: 'success',
                flex: 1,
                renderer: me.successRenderer
            }, {
                xtype: 'actioncolumn',
                width: 35,
                items: [{
                    iconCls: 'sprite-sticky-notes-pin',
                    action: 'openOrder',
                    tooltip: me.snippets.rows.openOrderTooltip,
                    handler: function(view, rowIndex, colIndex, item) {
                        var subApp = me.up('order-viison-shipping-common-batch-label-window').subApplication;
                        var orderId = view.getStore().getAt(rowIndex).get('orderId');
                        var orderStore = subApp.getStore('Order');
                        var orderRecord = orderStore.findRecord('id', orderId);
                        var detailController = subApp.getController('Detail');
                        detailController.onShowDetail(orderRecord);
                    }
                }]
            }
        ];
    },

    /**
     * Column renderer for the tracking code column.
     *
     * @param value The value to render.
     * @return The rendered value.
     */
    trackingCodeRenderer: function(value) {
        return value || this.subTextRenderer(this.snippets.rows.isNotTrackable);
    },

    /**
     * Column renderer for the success column.
     *
     * @param success The success flag.
     * @param metaData
     * @param record
     * @return The string corresponding to the batch label success flag.
     */
    successRenderer: function(success, metaData, record) {
        // Create the inner HTML elements for the cell
        var icon = {
            tag: 'span',
            class: this.statusIcons[(success) ? 0 : 1],
            style: 'margin: -2px 5px -2px 0; float: left;'
        };
        var text = {
            tag: 'span',
            style: 'display: table;',
            html: (success) ? this.snippets.rows.successMessage : record.get('message')
        };

        if (record.get('warningMessage')) {
            icon.class = this.warningIcon;
            text.html = record.get('warningMessage');
        }

        var helper = new Ext.dom.Helper;

        // Enable multi line column for only this row
        return '<div style="white-space:normal">' + helper.markup(icon) + helper.markup(text) + '</div>';
    },

    /**
     * Renders gray & italic text that can be used as a subtext or comment
     * @param string value
     */
    subTextRenderer: function (value) {
        var helper = new Ext.dom.Helper;
        var data = {
            tag: 'i',
            html: value,
            style: 'color: #7f8c8d'
        };

        return helper.markup(data);
    }

});
