// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

/**
 * The list containing all shipment documents of the active record. It contains four columns:
 *    - 'trackingCode', showing the tracking code/shipping number of the label
 *    - 'url', providing a link to the corresponding label PDF
 *    - 'created', showing the timestamp the label was created
 *    - a action column, holding the 'delete' button of each label
 */
//{namespace name="backend/viison_shipping_common_order/order"}
Ext.define('Shopware.apps.ViisonShippingCommonOrder.view.list.LabelShipment', {

    /**
     * Extend from the standard ExtJS 4.
     */
    extend: 'Ext.grid.Panel',

    /**
     * The item id used for the selector so we can create a backwards compatibility with the deprecated grid, example in refreshLabelGrid.
     * NOTE: itemId should be only used when we want to have a blind access to both grids
     */
    itemId: 'labelShipmentGrid',

    /**
     * List of short aliases for class names. Most useful for defining xtypes for widgets.
     */
    alias: 'widget.order-viison-shipping-common-label-shipment-list',

    /**
     * An optional extra CSS class that will be added to this component's Element.
     */
    cls: Ext.baseCSSPrefix + 'viison-shipping-common-label-grid',

    /**
     * The view needs to be scrollable.
     */
    autoScroll: true,

    /**
     * Delete label only local
     */
    onlyLocalDelete: false,

    /**
     * Url Action for getting document
     */
    url: '{url controller=ViisonShippingCommon action=getDocument}',

    /**
     * Contains all snippets for the view component.
     */
    snippets: {
        columns: {
            trackingCode: '{s name="label/columns/tracking_code"}{/s}',
            url: '{s name="label/columns/url"}{/s}',
            exportDocumentUrl: '{s name="label/columns/export_document_url"}{/s}',
            customerAddress: '{s name="label/columns/customer_address"}{/s}',
            type: '{s name="label/columns/type"}{/s}',
            created: '{s name="label/columns/created"}{/s}',
            trackLabel: '{s name="label/columns/track_label"}{/s}',
            deleteLabel: '{s name="label/columns/delete_label"}{/s}'
        },
        rows: {
            labelUrlTitle: '{s name="label/rows/label_url_title"}{/s}',
            exportDocumentUrlTitle: '{s name="label/rows/export_document_url_title"}{/s}',
            noExportDocumentUrlTitle: '{s name="label/rows/no_export_document_url_title"}{/s}',
            isShippingLabel: '{s name="label/rows/is_shipping_label"}{/s}',
            isReturnLabel: '{s name="label/rows/is_return_label"}{/s}',
            isNotTrackable: '{s name="label/rows/is_not_trackable"}{/s}',
            noDocumentUrlTitle: '{s name="label/rows/no_document_url_title"}{/s}',
            documentUrlTitle: '{s name="label/rows/document_url_title"}{/s}'
        }
    },

    /**
     * The main initializer registering events and adding view components.
     */
    initComponent: function() {
        this.columns = this.getColumns();

        this.dockedItems = [Ext.create('Ext.toolbar.Paging', {
            xtype: 'pagingtoolbar',
            store: this.store,
            dock: 'bottom',
            displayInfo: true
        })];

        this.callParent(arguments);
    },

    /**
     * Creates the grid columns.
     *
     * @return array The column structure.
     */
    getColumns: function() {
        var me = this;

        var columns = [
            {
                header: me.snippets.columns.trackingCode,
                dataIndex: 'trackingCode',
                flex: 7,
                renderer: me.trackingCodeRenderer
            }
        ];

        // Concat the documents columns
        columns = columns.concat(me.createDocumentLinksColumns());

        // It is used by free form labels
        if (me.showCustomerAddress) {
            columns.push(
                {
                    header: me.snippets.columns.customerAddress,
                    dataIndex: 'customerAddress',
                    flex: 10,
                    renderer: me.customerAddressRenderer
                }
            );
        }

        columns = columns.concat([
            {
                header: me.snippets.columns.type,
                dataIndex: 'returnShipment',
                flex: 5,
                renderer: me.typeRenderer
            },
            {
                header: me.snippets.columns.created,
                dataIndex: 'created',
                flex: 6,
                renderer: me.createdRenderer
            },
            me.createActionColumn()
        ]);

        return columns;
    },

    /**
     * Create document links columns.
     *
     * @return Array
     */
    createDocumentLinksColumns: function() {
        return [{
                header: this.snippets.columns.url,
                dataIndex: 'documents',
                flex: 10,
                renderer: function (documents, style, record) {
                    return this.renderDocumentUrl(documents, style, record, record.get('returnShipment') ? 'returnLabel' : 'label');
                }
            }];
    },

    /**
     * Creates a new column containing action icons.
     *
     * @return The grid action column.
     */
    createActionColumn: function() {
        var me = this;

        return Ext.create('Ext.grid.column.Action', {
            width: 60,
            items: [
                me.createTrackLabelColumn(),
                me.createDeleteLabelColumn()
            ]
        });
    },

    /**
     * Creates a new icon which opens a new tab/window showing the tracking of the label
     * in whose row the icon is clicked.
     *
     * @return The created icon.
     */
    createTrackLabelColumn: function() {
        var me = this;

        return {
            iconCls: 'c-sprite-box-search-result',
            action: 'trackShippingLabel',
            tooltip: me.snippets.columns.trackLabel,
            handler: function (view, rowIndex) {
                me.fireEvent('trackShippingLabel', view.getStore().getAt(rowIndex));
            },
            getClass: function (html, metadata, record) {
                // Hide this action icon if tracking code is not presented
                if (!record.raw.trackingCode) {
                    return 'x-hide-display';
                }
            },
        };
    },

    /**
     * Creates a new icon which triggers the deletion of the label
     * in whose row the icon is clicked.
     *
     * @return The created icon.
     */
    createDeleteLabelColumn: function() {
        var me = this;
        return {
            iconCls: 'sprite-minus-circle-frame',
            action: 'destroyShippingLabel',
            tooltip: me.snippets.columns.deleteLabel,
            handler: function (view, rowIndex) {
                me.fireEvent('destroyShippingLabel', me.dispatchServiceProviderPanel, me.record, view.getStore().getAt(rowIndex), view.getStore(), me.onlyLocalDelete);
            },
        };
    },

    /**
     * Column renderer for the tracking code column.
     *
     * @param string trackingCode The value to render.
     * @return The rendered value, if value doesn't exist render isNotTrackable text.
     */
    trackingCodeRenderer: function(trackingCode) {
        return trackingCode || this.subTextRenderer(this.snippets.rows.isNotTrackable);
    },

    /**
     * Render the document url title in the Grid.
     *
     * NOTE: The snippet should be used in format: no + TYPE + UrlTitle and TYPE + UrlTitle,
     *       if we want to use a custom snippet instead of the default one.
     *
     * @param documents The additional documents for the order entry without the parent label.
     * @param style Ext js grid style param
     * @param record The Store record.
     * @param documentType The document type to render.
     * @return string A HTML 'a' tag linking to the URL.
     */
    renderDocumentUrl: function (documents, style, record, documentType) {
        var document = documents.find(function (document) {
             return document.type === documentType;
        });

        // If document type is not present
        if (!document) {
            return this.snippets.rows['no' + documentType[0].toUpperCase() + documentType.substring(1) + 'UrlTitle'] || this.snippets.rows.noDocumentUrlTitle
        }

        return this.htmlLinkRenderer(this.createDocumentActionUrl(document.id, documentType), this.snippets.rows[documentType + 'UrlTitle'] || this.snippets.rows.documentUrlTitle);
    },

    /**
     * Column renderer for the receiver name column.
     *
     * @param object customerAddress The value to render.
     * @return string The rendered value.
     */
    customerAddressRenderer: function(customerAddress) {
        return customerAddress;
    },

    /**
     * Column renderer for the type column.
     *
     * @param bool returnShipment The value to render.
     * @return string The corresponding snippet for either the 'normal' or 'return' label.
     */
    typeRenderer: function(returnShipment) {
        return returnShipment ? this.snippets.rows.isReturnLabel : this.snippets.rows.isShippingLabel;
    },

    /**
     * Column renderer for the creation date column.
     *
     * @param string created The value to render.
     * @return DateTime A formatted date version of the input value.
     */
    createdRenderer: function(created) {
        if (created === Ext.undefined) {
            return created;
        }

        return Ext.util.Format.date(created, 'd.m.Y H:i:s');
    },

    /**
     * Creates a rendered html 'a' tag using the given URL and title.
     *
     * @param url The URL used as the href of the created 'a' tag.
     * @param title The text used as the label of the 'a' tag.
     * @return A HTML a-tag linking to the given URL.
     */
    htmlLinkRenderer: function(url, title) {
        var helper = new Ext.dom.Helper;
        var data = {
            tag: 'a',
            html: title,
            href: url,
            target: '_blank'
        };

        return helper.markup(data);
    },

    /**
     * Renders gray & italic text that can be used as a subtext or comment
     * @param string value
     */
    subTextRenderer: function (value) {
        var helper = new Ext.dom.Helper;
        var data = {
            tag: 'i',
            html: value,
            style: 'color: #7f8c8d'
        };

        return helper.markup(data);
    },

    /**
     * Create getDocumentAction url with params.
     *
     * NOTE: The $shipmentParamGetDocumentId and $shipmentParamGetDocumentType are called via smarty and defined in getViewParams.
     *
     * @param int identifier The document id.
     * @param string type The document type (export document, label, ...)
     * @return string A composed URL.
     */
    createDocumentActionUrl: function (identifier, type) {
        return this.url + '?{$shipmentParamGetDocumentId}=' + identifier +
                '&{$shipmentParamGetDocumentType}=' + type;
    }

});
