<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonSetArticles\Components;

use Shopware\Components\Model\ModelManager;
use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonSetArticles\Components\SetArticleOrderDetailAssociation\SetArticleOrderDetailAssociationDescription;

class SetArticleOrderDetailAssociationService
{
    /**
     * @var ModelManager $entityManager
     */
    protected $entityManager;

    public function __construct(ModelManager $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    /**
     * Assigns the OrderDetail attribute ViisonSetArticleOrderId to set- and sub articles after an order has been
     * created. Constructs the set connections from the temporary order detail attribute in the given $orderData and
     * persists the set information in the actual OrderDetail entities.
     *
     * @param Order $order already persisted order entity
     * @param SetArticleOrderDetailAssociationDescription[] $setArticleOrderDetailAssociations
     */
    public function associateSetArticleOrderDetails(Order $order, array $setArticleOrderDetailAssociations)
    {
        // Make association OrderDetailId -> SetArticleOrderNumber (possibly null) from the given order detail data
        $setArticleNumberByOrderDetailId = [];
        foreach ($setArticleOrderDetailAssociations as $setArticlePositionRelation) {
            $setArticleNumberByOrderDetailId[$setArticlePositionRelation->getOrderDetailId()] = $setArticlePositionRelation->getSetArticleNumber();
        }

        // Make association OrderDetailOrderNumber -> OrderDetail of easier access
        // We are aware that multiple sub articles in an order have the same article number. But since this array is
        // needed to find set article positions (which are unique in an order) this is no issue.
        $orderDetailByArticleNumber = [];
        foreach ($order->getDetails() as $orderDetail) {
            $orderDetailByArticleNumber[$orderDetail->getArticleNumber()] = $orderDetail;
        }

        // Combine sub article OrderDetail to set article OrderDetail and persist attributes
        $changedEntities = [];
        foreach ($order->getDetails() as $orderDetail) {
            $setArticleOrderNumber = $setArticleNumberByOrderDetailId[$orderDetail->getId()];
            if ($setArticleOrderNumber !== null &&
                $orderDetailByArticleNumber[$setArticleOrderNumber] !== null
            ) {
                $setArticleOrderDetailId = $orderDetailByArticleNumber[$setArticleOrderNumber]->getId();
                $orderDetail->getAttribute()->setViisonSetarticleOrderid($setArticleOrderDetailId);
                $changedEntities[] = $orderDetail->getAttribute();
            }
        }
        $this->entityManager->flush($changedEntities);
    }
}
