<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonSetArticles;

use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonSetArticles\Util;
use Doctrine\ORM\Mapping as ORM;

/**
 * SetArticle database layout object.
 *
 * A set article is an article that consists of multiple other articles (sub articles). Each sub article has an
 * individual quantity. Each Subsub articleArticle is linked to its set article via SetId (= ArticleDetailId of
 * set article).
 *
 * @ORM\Table(name="s_articles_viison_setarticles")
 * @ORM\Entity(repositoryClass="Repository")
 */
class SetArticle extends ModelEntity
{

    /**
     * @var int $id
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * Remark: setId is equal to the ArticleDetailId of the main set article.
     *
     * @var int $setId
     *
     * @ORM\Column(name="setid", type="integer", nullable=false)
     */
    private $setId;

    /**
     * @var \Shopware\Models\Article\Detail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="setid", referencedColumnName="id")
     */
    protected $setArticleDetail;

    /**
     * @var int $articleDetailId
     *
     * @ORM\Column(name="articledetailid", type="integer", nullable=false)
     */
    private $articleDetailId;

    /**
     * @var \Shopware\Models\Article\Detail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="articledetailid", referencedColumnName="id")
     */
    protected $articleDetail;

    /**
     * @var int quantity
     *
     * @ORM\Column(name="quantity", type="integer", nullable=false)
     */
    private $quantity;

    /**
     * @return \Shopware\Models\Article\Detail
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @param \Shopware\Models\Article\Detail $articleDetail
     */
    public function setArticleDetail($articleDetail)
    {
        $this->articleDetail = $articleDetail;
    }

    /**
     * @return \Shopware\Models\Article\Detail
     */
    public function getSetArticleDetail()
    {
        return $this->setArticleDetail;
    }

    /**
     * @param \Shopware\Models\Article\Detail $setArticleDetail
     */
    public function setSetArticleDetail($setArticleDetail)
    {
        $this->setArticleDetail = $setArticleDetail;
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param int $id
     */
    public function setId($id)
    {
        $this->id = $id;
    }

    /**
     * @return int
     */
    public function getSetId()
    {
        return $this->setId;
    }

    /**
     * @param int $newSetId
     */
    public function setSetId($newSetId)
    {
        $this->setId = $newSetId;
    }

    /**
     * @return int
     */
    public function getArticleDetailId()
    {
        return $this->articleDetailId;
    }

    /**
     * @param int $newArticleDetailId
     */
    public function setArticleDetailId($newArticleDetailId)
    {
        $this->articleDetailId = $newArticleDetailId;
    }

    /**
     * @return int
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * @param int $quantity
     */
    public function setQuantity($quantity)
    {
        $this->quantity = $quantity;
    }

    /**
     * Returns the fully assembled sub article name which consists of the set article prefix, set article ordernumber,
     * sub article name and additional.
     *
     * @param Shop|null $shop
     * @return string
     */
    public function getSubArticleName($shop = null)
    {
        $setArticlePrefix = Util::getSubArticleNamePrefix(
            $this->setArticleDetail->getNumber(),
            $shop
        );
        $articleName = Util::getTranslatedFullArticleName($this->articleDetail, $shop);
        $result = $setArticlePrefix . ': ' . $articleName;

        return $result;
    }
}
