<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonSetArticles\Subscribers;

use Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\CompatibilityCheck as ViisonCommonCompatibilityCheckSubscriber;

class CompatibilityCheckSubscriber extends ViisonCommonCompatibilityCheckSubscriber
{
    /**
     * @inheritdoc
     */
    public function getPluginCompatibilityConstraints()
    {
        $snippets = $this->get('snippets')->getNamespace('backend/viison_set_articles/compatibilitycheck');

        return [
            $this->getMagnalisterConstraint(),
            $this->getPickwareERPGrossProfitConstraint(),
            new PluginCompatibilityConstraint(
                'Frontend',
                'HeidelGateway',
                '9999.9999.9999', // mark as "incompatible"
                $snippets->get('HeidelGateway')
            ),
        ];
    }

    /**
     * Creates and returns the MagnalisterCompatibilityConstraint
     *
     * @return \Shopware\Plugins\ViisonCommon\Structs\PluginCompatibilityConstraint
     */
    private function getMagnalisterConstraint()
    {
        $magnalisterContraint = new PluginCompatibilityConstraint();

        $magnalisterContraint->setCustomMessage(
            $this->get('snippets')->getNamespace('backend/viison_set_articles/main')->get(
                'compatibilitycheck/magnalister/fail'
            )
        );
        $magnalisterContraint->setLink(
            'https://docs.google.com/document/d/15gFRkR8ZCdTPJPfQBLX8oNAv5c1ZqgvdNQYG-VeM2u8/edit#heading=h.gk5mn3wi1vf1'
        );
        $magnalisterContraint->setLinkTitle(
            $this->get('snippets')->getNamespace('backend/viison_set_articles/main')->get(
                'compatibilitycheck/documentation'
            )
        );
        $magnalisterHookCheck = function () {
            // First check if Magnalister is installed and active. If not: return true since no patch is needed
            if (!ViisonCommonUtil::isPluginInstalledAndActive('Backend', 'RedMagnalister')) {
                return true;
            }

            // swroot/engine/Shopware/Plugins/Community/Backend/RedMagnalister/Lib/Codepool/10_Customer/Hooks/Hook/addOrder_X.php
            $dir = $this->getPluginPath() . '../../../Community/Backend/RedMagnalister/Lib/Codepool/10_Customer/Hooks/Hook/';
            $filePattern = 'addOrder*.php';
            $files = glob($dir . $filePattern);
            if (empty($files)) {
                return true;
            } elseif (count($files) == 1) {
                $hookKeyWord = '* Magnalister - SetArticles';

                // If the hook file contains set articles logic (strpos !== false), it has been patched - return true
                return (mb_strpos(file_get_contents($files[0]), $hookKeyWord) !== false);
            } else {
                // More than one hook file - this cannot work - return false (and alert customer)
                return false;
            }
        };
        $magnalisterContraint->setCustomCheck($magnalisterHookCheck);

        return $magnalisterContraint;
    }

    /**
     * Returns a constraint for ViisonPickwareERP version 3.2.54.
     * Since much refactoring was done in the GrossProfitAnalytics controller and subscriber, both plugin
     * need to be at a certain version to work together without throwing exceptions.
     *
     * @return PluginCompatibilityConstraint
     */
    private function getPickwareERPGrossProfitConstraint()
    {
        $grossProfitConstraint = new PluginCompatibilityConstraint();
        $grossProfitConstraint->setModule('Core');
        $grossProfitConstraint->setPlugin('ViisonPickwareERP');
        $grossProfitConstraint->setMinimumVersion('3.2.54');
        $grossProfitConstraint->setLink('https://docs.google.com/document/d/15gFRkR8ZCdTPJPfQBLX8oNAv5c1ZqgvdNQYG-VeM2u8/edit');

        return $grossProfitConstraint;
    }
}
