<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonSetArticles\Subscribers;

use Enlight_Event_EventArgs;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\AbstractBaseSubscriber;
use Shopware\Plugins\ViisonSetArticles\Components\ConfiguratorGatewayDecorator\ConfiguratorGatewayDecoratorPHP5;
use Shopware\Plugins\ViisonSetArticles\Components\ConfiguratorGatewayDecorator\ConfiguratorGatewayDecoratorPHP7;
use Shopware\Plugins\ViisonSetArticles\Components\ListProductGatewayDecorator;
use Shopware\Plugins\ViisonSetArticles\Components\ProductNumberServiceDecorator;
use Shopware\Plugins\ViisonSetArticles\Components\QueryBuilderFactoryDecorator;

class ServiceDecorationSubscriber extends AbstractBaseSubscriber
{
    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Enlight_Bootstrap_AfterInitResource_shopware_storefront.list_product_gateway' => 'onAfterInitListProductGateway',
            'Enlight_Bootstrap_AfterInitResource_shopware_storefront.product_number_service' => 'onAfterInitProductNumberService',
            'Enlight_Bootstrap_AfterInitResource_shopware_storefront.configurator_gateway' => 'onAfterInitConfiguratorGateway',
            'Enlight_Bootstrap_AfterInitResource_shopware_searchdbal.dbal_query_builder_factory' => 'onAfterInitQueryBuilderFactory',
        ];
    }

    /**
     * Decorates the 'shopware_storefront.list_product_gateway' service which is used as a source for product (article)
     * information in many parts of the store frontend. Our decorator service manipulates set article availability.
     *
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterInitListProductGateway(Enlight_Event_EventArgs $args)
    {
        $args->get('subject')->set(
            'shopware_storefront.list_product_gateway',
            new ListProductGatewayDecorator(
                $args->get('subject')->get('shopware_storefront.list_product_gateway'),
                $this->get('models')
            )
        );
    }

    /**
     * Decorates the `shopware_storefront.product_number_service` service which manipulates the availability check of
     * article (numbers) in the frontend.
     *
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterInitProductNumberService(Enlight_Event_EventArgs $args)
    {
        $args->get('subject')->set(
            'shopware_storefront.product_number_service',
            new ProductNumberServiceDecorator(
                $args->get('subject')->get('shopware_storefront.product_number_service'),
                $this->get('models')
            )
        );
    }

    /**
     * Decorates the 'shopware_storefront.configurator_gateway' service which is used as a source for variant
     * group/options mapping. Our decorator service manipulates the availability of set article options.
     *
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterInitConfiguratorGateway(Enlight_Event_EventArgs $args)
    {
        if (PHP_MAJOR_VERSION < 7) {
            $args->get('subject')->set(
                'shopware_storefront.configurator_gateway',
                new ConfiguratorGatewayDecoratorPHP5(
                    $args->get('subject')->get('shopware_storefront.configurator_gateway'),
                    $this->get('models')
                )
            );
        } else {
            $args->get('subject')->set(
                'shopware_storefront.configurator_gateway',
                new ConfiguratorGatewayDecoratorPHP7(
                    $args->get('subject')->get('shopware_storefront.configurator_gateway'),
                    $this->get('models')
                )
            );
        }
    }

    /**
     * Decorates the 'shopware_searchdbal.dbal_query_builder_factory' service which creates a product number search
     * query that is used for article retrieval and filtering in the store frontend. Our decorator service manipulates
     * the query to enable the immediate delivery filter for set articles.
     *
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterInitQueryBuilderFactory(\Enlight_Event_EventArgs $args)
    {
        $args->get('subject')->set(
            'shopware_searchdbal.dbal_query_builder_factory',
            new QueryBuilderFactoryDecorator(
                $args->get('subject')->get('shopware_searchdbal.dbal_query_builder_factory')
            )
        );
    }
}
