<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document\RenderingContext;

use Shopware\Plugins\ViisonCommon\Classes\Document\GridPaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Document\PaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Document\RenderingEngine\RenderingEngine;

/**
 * This is a rendering context for rendering grid layout based documents.
 *
 * That are documents which have a certain number of rows and columns.
 */
class GridLayoutRenderingContext extends TemplateRenderingContext
{
    /**
     * @var mixed[] The contents of the cells
     */
    protected $items = [];

    /**
     * @var PaperLayout
     */
    protected $gridPaperLayout;

    /**
     * @var int
     */
    protected $offsetX;

    /**
     * @var int
     */
    protected $offsetY;

    /**
     * @param RenderingEngine $renderingEngine
     * @param \Enlight_Template_Manager $templateManager
     * @param GridPaperLayout $gridPaperLayout
     * @param string $templateFilePath The full path of the template file to be rendered
     */
    public function __construct(
        RenderingEngine $renderingEngine,
        \Enlight_Template_Manager $templateManager,
        GridPaperLayout $gridPaperLayout,
        $templateFilePath
    ) {
        $this->gridPaperLayout = $gridPaperLayout;
        $templateDir = dirname($templateFilePath);
        $templateFilePath = 'extends:GridLayoutWithAbsolutePositioning.tpl|' . $templateFilePath;

        parent::__construct($renderingEngine, $templateManager, $templateFilePath, $gridPaperLayout);

        // Add the path of ShopwareCommon where the templates for all grid layouts are stored to make all layouts
        // available.
        $this->addTemplateDir(__DIR__ . '/../../../Views/documents');

        // Add template directory of the template file to avoid some nasty template security problems.
        $this->addTemplateDir($templateDir);
    }

    /**
     * @inheritdoc
     */
    protected function generateHtml()
    {
        $emptyItemCount = $this->offsetY * $this->gridPaperLayout->columnAmount + $this->offsetX;
        // PHP 5.5 and lower does not support 0 as second argument for array_fill
        if ($emptyItemCount !== 0) {
            $items = array_merge(array_fill(0, $emptyItemCount, null), $this->items);
        } else {
            $items = $this->items;
        }
        $this->assignTemplateVar('items', $items);
        $this->assignTemplateVar('layout', $this->gridPaperLayout);

        return parent::generateHtml();
    }

    /**
     * @return array
     */
    public function getItems()
    {
        return $this->items;
    }

    /**
     * @param array $items An array of items, which will be rendered. The content of the array is arbitrary.
     */
    public function setItems(array $items)
    {
        $this->items = $items;
    }

    /**
     * @param array $item Add a item to items that should be renderer
     */
    public function addItem($item)
    {
        $this->items[] = $item;
    }

    /**
     * @return int
     */
    public function getOffsetX()
    {
        return $this->offsetX;
    }

    /**
     * @param int $offsetX
     */
    public function setOffsetX($offsetX)
    {
        $this->offsetX = $offsetX;
    }

    /**
     * @return int
     */
    public function getOffsetY()
    {
        return $this->offsetY;
    }

    /**
     * @param int $offsetY
     */
    public function setOffsetY($offsetY)
    {
        $this->offsetY = $offsetY;
    }

    /**
     * @return PaperLayout
     */
    public function getGridPaperLayout()
    {
        return $this->gridPaperLayout;
    }

    /**
     * @param PaperLayout $gridPaperLayout
     */
    public function setGridPaperLayout($gridPaperLayout)
    {
        $this->gridPaperLayout = $gridPaperLayout;
        parent::setPaperLayout($gridPaperLayout);
    }
}
