<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonCommon\Classes\Document\RenderingContext;

use Shopware\Plugins\ViisonCommon\Classes\Document\PaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Document\RenderingEngine\RenderingEngine;

/**
 * RenderingContext to render a smarty template given by its path.
 */
class TemplateRenderingContext extends AbstractRenderingContext
{
    /**
     * @var array Associative array of template variables, which will be passed to the template, when it is rendered
     */
    protected $templateVars = [];

    /**
     * @var  \Enlight_Template_Manager
     */
    protected $templateManager;

    /**
     * @var string Absolute or relative path to the smarty template to be rendered
     */
    protected $templateFilePath;

    /**
     * @param RenderingEngine $renderingEngine
     * @param \Enlight_Template_Manager $templateManager
     * @param string $templateFilePath Absolute or relative path to the smarty template to be rendered
     * @param $paperLayout $paperLayout
     */
    public function __construct(
        RenderingEngine $renderingEngine,
        \Enlight_Template_Manager $templateManager,
        $templateFilePath,
        PaperLayout $paperLayout
    ) {
        parent::__construct($renderingEngine, $paperLayout);
        $this->templateManager = $templateManager;
        $this->templateFilePath = $templateFilePath;
    }

    /**
     * @inheritdoc
     */
    protected function generateHtml()
    {
        $template = $this->templateManager->createData();
        $template->assign($this->getTemplateVars());

        // Add template directory of the template file to avoid some nasty template security problems.
        $this->addTemplateDir(dirname($this->templateFilePath));

        return $this->templateManager->fetch($this->templateFilePath, $template);
    }

    /**
     * @return array
     */
    public function getTemplateVars()
    {
        return $this->templateVars;
    }

    /**
     * @param array $templateVars
     */
    public function setTemplateVars(array $templateVars)
    {
        $this->templateVars = $templateVars;
    }

    /**
     * Adds one or more template variables at once
     *
     * @param string|array $nameOrArray
     * @param mixed $value
     */
    public function assignTemplateVar($nameOrArray, $value = null)
    {
        if (is_array($nameOrArray)) {
            $this->templateVars = array_merge($this->templateVars, $nameOrArray);
        } else {
            $this->templateVars[$nameOrArray] = $value;
        }
    }

    /**
     * @return \Enlight_Template_Manager
     */
    public function getTemplateManager()
    {
        return $this->templateManager;
    }

    /**
     * @param \Enlight_Template_Manager $templateManager
     */
    public function setTemplateManager(\Enlight_Template_Manager $templateManager)
    {
        $this->templateManager = $templateManager;
    }

    /**
     * @return string
     */
    public function getTemplateFilePath()
    {
        return $this->templateFilePath;
    }

    /**
     * @param string $templateFilePath
     */
    public function setTemplateFilePath($templateFilePath)
    {
        $this->templateFilePath = $templateFilePath;
    }

    /**
     * Adds a directory to the template search path
     *
     * @param string|array $templateDir directory(s) of template sources
     * @param string $key of the array element to assign the template dir to
     * @param string $position
     */
    public function addTemplateDir($templateDir, $key = null, $position = \Enlight_Template_Manager::POSITION_PREPEND)
    {
        $this->templateManager->addTemplateDir($templateDir, $key, $position);
    }
}
