<?php
namespace Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\SwagImportExportVersion2;

/**
 * Copy of Shopware SwagImportExport Validator in Version 2.12.1
 */
abstract class SwagAdapterValidator
{
    /**
     * Removes fields which contain empty string.
     *
     * @param array $record
     *
     * @return array
     */
    public function filterEmptyString($record)
    {
        return \array_filter($record, 'strlen');
    }

    /**
     * Validates fields types.
     *
     * @param array $record
     * @param array $mapper
     *
     * @throws SwagAdapterException
     * @throws \Exception
     */
    public function validate($record, $mapper)
    {
        foreach ($record as $fieldName => $value) {
            foreach ($mapper as $type => $fields) {
                if (\in_array($fieldName, $fields)) {
                    $this->validateType($type, $value, $fieldName);
                    break;
                }
            }
        }
    }

    /**
     * Validates fields with int type. It is possible this field to has as a value '-1'.
     *
     * @return int
     */
    public function validateInt($value)
    {
        return \preg_match('/^-{0,1}\\d+$/', $value);
    }

    /**
     * Validates fields with float type.
     *
     * @param string $value
     *
     * @return int
     */
    public function validateFloat($value)
    {
        return \preg_match('/^-?\\d+((\\.|,){0,1}\\d+)*$/', $value);
    }

    /**
     * Validates fields which contains date data.
     *
     * @return bool
     */
    public function validateDateTime($value)
    {
        return (bool) \strtotime($value);
    }

    /**
     * Validates email fields.
     *
     * @throws \Exception
     */
    public function validateEmail($email)
    {
        /** @var \Shopware\Components\Validator\EmailValidatorInterface $emailValidator */
        $emailValidator = Shopware()->Container()->get('validator.email');

        return $emailValidator->isValid($email);
    }

    /**
     * Validates fields which contains string.
     *
     * @return bool
     */
    public function validateString($value)
    {
        return \is_string($value);
    }

    /**
     * Helper function, which is used to validate current field's value.
     *
     * @param string $type
     * @param string $fieldName
     *
     * @throws SwagAdapterException
     * @throws \Exception
     */
    private function validateType($type, $value, $fieldName)
    {
        $action = 'validate' . \ucfirst($type);
        if (!\is_callable([$this, $action])) {
            throw new \Exception('Method with name `' . $action . '` does not exist!');
        }

        $isCorrect = $this->$action($value);

        if (!$isCorrect) {
            $message = SwagSnippetsHelper::getNamespace()->get('validators/wrong_type', '%s field has to be %s but is %s!');
            throw new SwagAdapterException(\sprintf($message, $fieldName, $type, $value));
        }
    }
}
