// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonCommonPaginationToolbar.view.Toolbar', {

    extend: 'Ext.toolbar.Paging',
    alias: 'widget.viison_common_pagination_toolbar-toolbar',
    cls: 'viison_common_pagination_toolbar-toolbar',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    dock: 'bottom',
    displayInfo: true,
    fallbackPageSize: 100,
    paginationSteps: [50, 100, 250, 500],
    viisonSnippetNamespace: 'backend/viison_common_pagination_toolbar/main',

    /**
     * @override
     */
    constructor: function () {
        this.subject = this.getViisonSnippet('subject');

        this.callParent(arguments);
    },

    initComponent: function () {
        this.callParent();
        this.paginationSteps = this.paginationSteps.slice();

        this.ensurePaginationStepsContainDefaultPagesize();
        if (this.shouldAddPageSizeSelection()) {
            this.addPageSizeSelection();
        }
    },

    /**
     * The default page size that should originate from the store should be selectable by the page size selection
     */
    ensurePaginationStepsContainDefaultPagesize: function () {
        var paginationStepsContainsDefaultPageSize = this.paginationSteps
            && (this.paginationSteps.indexOf(this.getDefaultPageSize()) !== -1);

        if (!paginationStepsContainsDefaultPageSize) {
            // Ensure paginationSteps is an array
            if (!this.paginationSteps) {
                this.paginationSteps = [];
            } else {
                this.paginationSteps.push(this.getDefaultPageSize());
            }
        }
    },

    /**
     * Decides if page size selection should be displayed
     * @return boolean
     */
    shouldAddPageSizeSelection: function () {
        return this.paginationSteps && this.paginationSteps.length > 1;
    },

    /**
     * Look in different places for a default page size
     * @return number
     */
    getDefaultPageSize: function () {
        return this.store.pageSize
            || (this.paginationSteps ? this.paginationSteps[0] : undefined)
            || this.fallbackPageSize;
    },

    /**
     * Create combo box for page size selection
     */
    addPageSizeSelection: function () {
        var pageSizeSelection = Ext.create('Ext.form.field.ComboBox', {
            width: 186,
            labelWidth: 120,
            fieldLabel: this.subject + ' ' + this.getViisonSnippet('subjects_per_page'),
            cls: Ext.baseCSSPrefix + 'page-size',
            displayField: 'value',
            valueField: 'value',
            queryMode: 'local',
            store: this.getPaginationStore(),
        });

        this.add(
            '->',
            pageSizeSelection,
            {
                xtype: 'tbspacer',
                width: 6,
            }
        );

        pageSizeSelection.setValue(this.getDefaultPageSize());
        pageSizeSelection.on('change', this.onChangePageSize, this);
    },

    /**
     * Creates the store for the combo box
     * @return Ext.data.Store
     */
    getPaginationStore: function () {
        var sortedPaginationSteps = this.paginationSteps.sort(function (a, b) {
            return a - b;
        });

        return Ext.create('Ext.data.Store', {
            fields: [
                'value',
            ],
            data: Ext.Array.map(sortedPaginationSteps, function (i) {
                return {
                    value: i,
                };
            }),
        });
    },

    /**
     * Uses the new value as the page size of the store and reloads its data.
     *
     * @param combo The combobox firing the event.
     * @param newValue The new value of the combobox.
     */
    onChangePageSize: function (combo, newValue) {
        this.store.pageSize = (newValue || this.getDefaultPageSize());
        this.store.currentPage = 1;
        this.store.load();
    },

});
