<?php

/**
 * Shopware 5
 * Copyright (c) shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 */

use Composer\Autoload\ClassLoader;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\DBAL\Connection;
use Shopware\Components\Random;
use Shopware\Plugins\SwagSecurity\Components\ShopwareVersion;
use Shopware\Plugins\SwagSecurity\Components\UpdateHtaccess;
use Shopware\Plugins\SwagSecurity\Components\Validator\EmailValidatorDecorator;
use Shopware\Plugins\SwagSecurity\Exception\SwagSecurityCSRFCheckoutException;
use Symfony\Component\HttpFoundation\Request;
use voku\helper\AntiXSS;

require_once __DIR__ . '/Components/CWE367/init.php';

class Shopware_Plugins_Core_SwagSecurity_Bootstrap extends Shopware_Components_Plugin_Bootstrap
{
    const ALLOWED_ATTRIBUTES_KEY = 'allowedAttributes';
    const ALLOWED_HTML_TAGS_KEY = 'allowedHtmlTags';

    /**
     * @var array<string, array<string, array<string, array<string>>>>
     *
     * usage:
     *
     * 'frontend/account/login' => [
     *      'password' => [
     *           self::ALLOWED_ATTRIBUTES_KEY => [],
     *           self::ALLOWED_HTML_TAGS_KEY => []
     *       ]
     *   ]
     */
    public $allowanceList = array();

    private $finishActionCalled = false;

    public function getCapabilities()
    {
        return array(
            'install' => true,
            'enable' => true,
            'update' => true,
            'secureUninstall' => false,
        );
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return 'Shopware Security Plugin';
    }

    /**
     * @return array
     */
    public function getInfo()
    {
        return array(
            'version' => $this->getVersion(),
            'label' => $this->getLabel(),
        );
    }

    /**
     * @throws Exception
     */
    public function getVersion()
    {
        $info = json_decode(file_get_contents(__DIR__ . '/plugin.json'), true);
        if ($info) {
            return $info['currentVersion'];
        }
        throw new \RuntimeException('The plugin has an invalid version file.');
    }

    public function install()
    {
        $this->addCacheDirectoryHtaccess();
        $this->deleteDeprecatedEvents();

        $this->registerController('Backend', 'SwagSecurityUpdate');

        $this->subscribeEvent('Enlight_Controller_Front_RouteShutdown', 'fixXSSNewsletter22065ExtendXssRegex', -400);
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Newsletter', 'fixXSSNewsletter22065ExtendTemplate');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_PluginManager', 'getPluginMetaData21776');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_PluginManager', 'fixMetaDataIntegrity21776');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Checkout', 'fixCSRFAddArticle20983');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Checkout', 'fixCSRFFinishCheckout20983');
        $this->subscribeEvent('Shopware_Controllers_Frontend_Checkout::saveOrder::before', 'fixCSRFFinishCheckoutSaveOrderBefore20983');
        $this->subscribeEvent('Enlight_Controller_Front_PostDispatch', 'fixCSRFFinishCheckoutPostDispatch20983');

        $this->subscribeEvent('Shopware_Controllers_Frontend_Checkout::getBasket::after', 'fixUnallowedCurrencyChangeGetBasket21593');
        $this->subscribeEvent('Shopware_Modules_Order_SaveOrder_FilterParams', 'fixUnallowedCurrencyChangeFilterParams21593');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Newsletter', 'fixXSSNewsletter20878');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Address', 'fixXSSAddress20878');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Account', 'fixRecoverPassword21640');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_ProductStream', 'new_fixSQLi_19895');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_ProductStream', 'new_fixXXE_19896');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_ProductStream', 'fix_xxe_in_product_stream_24068');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Order', 'new_fixXSS_19834');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Customer', 'new_fixXSS_19834');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Address', 'new_fixXSS_19834');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_ImportExport', 'fixExportOutput');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Mail', 'fixMailSecurity');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Widgets', 'fixWidgetsSecurity');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Attributes', 'fixSQLSecurity_21404');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Emotion', 'fixPathTraversal21412');

        $this->subscribeEvent('Shopware_Controllers_Backend_Order::getOrderDocumentsQuery::before', 'getOrderDocumentsQuery');
        $this->subscribeEvent('Shopware\Models\Article\Repository::getConfiguratorSetsWithExcludedIdsQueryBuilder::before', 'fixIdsArray');
        $this->subscribeEvent('Shopware\Models\Article\Repository::getArticlesWithExcludedIdsQueryBuilder::before', 'fixIdsArray');
        $this->subscribeEvent('Shopware\Models\Article\Repository::getSuppliersWithExcludedIdsQueryBuilder::before', 'fixIdsArray');
        $this->subscribeEvent('Shopware\Models\Blog\Repository::getListQueryBuilder::before', 'fixBlogRepositoryListQuery');
        $this->subscribeEvent('Shopware\Models\Blog\Repository::getFilterQueryBuilder::before', 'getFilterQueryBuilderBlogRepository');
        $this->subscribeEvent('Shopware\Models\Blog\Repository::getBackendListQueryBuilder::before', 'getBackendListQueryBuilderBlogRepository');
        $this->subscribeEvent('Shopware\Models\Customer\Repository::getCustomerDetailQueryBuilder::before', 'fixCustomerIdParameter');
        $this->subscribeEvent('Shopware\Models\Customer\Repository::getOrdersQueryBuilder::before', 'fixCustomerIdParameter');
        $this->subscribeEvent('Shopware\Models\Dispatch\Repository::getShippingCostsMatrixQueryBuilder::before', 'fixDispatchIdParameter');
        $this->subscribeEvent('Shopware\Models\Dispatch\Repository::getPaymentQueryBuilder::before', 'fixUsedIdsFilter');
        $this->subscribeEvent('Shopware\Models\Dispatch\Repository::getCountryQueryBuilder::before', 'fixUsedIdsFilter');
        $this->subscribeEvent('Shopware\Models\Dispatch\Repository::getHolidayQueryBuilder::before', 'fixUsedIdsFilter');
        $this->subscribeEvent('Shopware\Models\Media\Repository::getAlbumMediaQueryBuilder::before', 'fixMediaIdParameter');
        $this->subscribeEvent('Shopware\Models\Order\Document\Repository::getListQuery::before', 'fixOrderIdParameter');
        $this->subscribeEvent('Shopware\Models\Voucher\Repository::getValidateOrderCodeQueryBuilder::before', 'fixVoucherIdParameter');
        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Api_Media', 'fixPathTraversal21151');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Media', 'fixPathTraversal23011');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Api_Media', 'fixPharDeserialization22386');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Article', 'fixEsdUploadSecurity23010');

        $this->subscribeEvent('Enlight_Controller_Front_PreDispatch', 'fixValidationBypass23012');

        $this->subscribeEvent('Shopware\Models\ProductFeed\ProductFeed::prePersist', 'fixProductFeed23009');
        $this->subscribeEvent('Shopware\Models\ProductFeed\ProductFeed::preUpdate', 'fixProductFeed23009');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Forms', 'fixXssOnForms23319');

        $this->subscribeEvent('Enlight_Controller_Front_PreDispatch', 'fixSessionFixation_23428');
        $this->subscribeEvent('Enlight_Controller_Action_Backend_Theme_preview', 'fixThemePreview_23428');
        $this->subscribeEvent('Enlight_Controller_Action_Frontend_Index_index', 'fixThemePreviewRestore_23428');
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Frontend', 'fixAjaxCalls_23428');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Backend_Customer', 'fixPerformOrder23016');

        $this->subscribeEvent('Theme_Compiler_Collect_Plugin_Javascript', 'onCollectJavaScriptFiles23007');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_Index', 'includeNotifierJavascript');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Frontend_Error', 'fixExceptionMessage23603');
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_Error', 'fixExceptionMessage23603');

        $this->subscribeEvent('sArticles::sCheckIfEsd::before', 'castParametersOnCheckEsd23766');

        $this->subscribeEvent('Enlight_Bootstrap_AfterInitResource_plugins', 'fixOperatorsForDql23626');
        $this->subscribeEvent('Enlight_Controller_Front_RouteShutdown', 'fixXSS_24473', 5);
        $this->subscribeEvent('Shopware_Controllers_Frontend_Register_DoubleOptIn_ResendMail', 'fixInvalidDOIUser_24590');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Blog', 'fixCachedBlogComment_25409');
        $this->subscribeEvent('Enlight_Controller_Dispatcher_ControllerPath_Widgets_Blog', 'registerController_25409');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_Customer', 'escapeCustomerMailInCustomerModule_25773');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_NewsletterManager', 'preventJavascriptExec_25772');
        $this->subscribeEvent('Shopware_Controllers_Backend_Newsletter::getMailingDetails::after', 'filterNewsletterTitle_25772');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Widgets_Emotion', 'preventXssInEmotionElements_25771');
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_Emotion', 'preventXssInYoutubeElement_25771');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Error', 'preventJsonResponseInErrorController_26001');

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_Index', 'adjustGetTextImplementation_26050');
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Backend_Article', 'escapeProductNameInProductModule_26050');
        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Frontend_Detail', 'escapeProductNameOnDetailPage_26050');

        $this->subscribeEvent('Enlight_Controller_Front_RouteShutdown', 'fixArbitraryRedirect_26435', -1);

        $this->subscribeEvent('Enlight_Controller_Front_RouteShutdown', 'fixArraysXss_26657');

        $this->subscribeEvent('Enlight_Controller_Front_RouteShutdown', 'fixXss_26748', -1);

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatch_Frontend_Account', 'addPasswordChangeDateToSessionUponPasswordChange_26448');
        $this->subscribeEvent('Shopware_Modules_Admin_Login_Successful', 'addPasswordChangeDateToSessionUponSuccessfulLogin_26448');

        $this->subscribeEvent('Shopware_Modules_Admin_Login_Successful', 'clearCsrfToken_26662');
        $this->subscribeEvent('sAdmin::logout::after', 'clearCsrfToken_26662');

        $this->subscribeEvent('Shopware_Modules_Admin_CheckUser_FilterGetUser', 'checkPasswordChangeDateInCheckUser_26448');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch_Frontend_Account', 'invalidateExistingTokensUponTokenCreation_26666');

        $this->subscribeEvent('Enlight_Controller_Action_PreDispatch', 'fixAclResource_26913', -1);

        $this->subscribeEvent('Enlight_Controller_Action_PostDispatchSecure_Backend_Customer', 'fixSensitiveData_26909');

        $this->subscribeEvent('Enlight_Bootstrap_AfterInitResource_validator.email', 'decorateEmailValidation');

        $form = $this->Form();
        $version = $this->swVersion();
        $majorVersion = (int) $this->swVersion('major');

        if (version_compare($version, UpdateHtaccess::getHtaccessChangeVersion(), '<')) {
            (new UpdateHtaccess())->update(
                Shopware()->Container()->getParameter('kernel.root_dir') . '/.htaccess',
                __DIR__ . '/Assets/.htaccess.dist'
            );
        }

        if (
            $majorVersion === 5
            && version_compare($version, '5.2.25', '<')
        ) {
            $form->setElement('checkbox', 'fix_export_output', array(
                'label' => 'Import/Export #18912',
                'description' => 'Escaping of the format request parameter',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_widgets_smarty_security', array(
                'label' => 'Smarty Security Mail #18917',
                'description' => 'Enable template engine security for mail verify smarty call',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_mail_smarty_security', array(
                'label' => 'Smarty Security Widgets #18917',
                'description' => 'Enable template engine security for send mail to merchant call',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_attribute_reset', array(
                'label' => 'Attribute system security #18997',
                'description' => 'Escaping of table and column name parameter',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_sql_injection', array(
                'label' => 'Possible sql injections #18915',
                'description' => 'Escaping different parameters in repositories',
                'value' => 1,
            ));
        }

        // Fix XSS in all versions > 5.0
        if (
            $majorVersion === 5
            && version_compare($version, '5.3.4', '<')
        ) {
            $form->setElement('checkbox', 'fix_xss_19834', array(
                'label' => 'Possible XSS injections #19834',
                'description' => 'Fix XSS injection in backend',
                'value' => 1,
            ));
        }

        // Fix ProductStream related issues only in the applicable versions
        if (
            version_compare($version, '5.1.0', '>=')
            && version_compare($version, '5.3.4', '<')
        ) {
            $form->setElement('checkbox', 'fix_sql_injection_19895', array(
                'label' => 'Possible backend sql injections #19895',
                'description' => 'Fix SQL injections in backend',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_xxe_19896', array(
                'label' => 'Possible backend XXE injection #19896',
                'description' => 'Fix XXE injection in backend',
                'value' => 1,
            ));
        }

        // Fix newsletter/address related xss issues only in the applicable versions
        if (
            version_compare($version, '5.2.0', '>=')
            && version_compare($version, '5.3.7', '<')
        ) {
            $form->setElement('checkbox', 'fix_xss_20878', array(
                'label' => 'Non persistent XSS #20878',
                'description' => 'Improve input validation for frontend parameter',
                'value' => 1,
            ));
        }

        // Fix Checkout CSRF related issues only in the applicable versions
        if (version_compare($version, '5.4.0', '<')) {
            $form->setElement('checkbox', 'fix_csrf_20938', array(
                'label' => 'Frontend checkout CSRF #20983',
                'description' => 'Activate further protection of the checkout process against CSRF attacks',
                'value' => 0,
            ));
        }

        // Fix attribute administration related issues only in the applicable versions
        if (
            version_compare($version, '5.1.0', '>=')
            && version_compare($version, '5.4.3', '<')
        ) {
            $form->setElement('checkbox', 'fix_sql_injection_21404', array(
                'label' => 'Possible backend sql injections #21404',
                'description' => 'Fix SQL injections in attribute administration in backend',
                'value' => 1,
            ));
        }

        // Fix path traversal media upload
        if (version_compare($version, '5.4.3', '<')) {
            $form->setElement('checkbox', 'fix_path_traversal_21151', array(
                'label' => 'Possible media upload path traversal #21151',
                'description' => 'Fix path traversal in resource API media',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_currency_change_21593', array(
                'label' => 'Fix unallowed currency change #21593',
                'description' => 'Fix the unallowed currency change while being on the order confirm page',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_path_traversal_21412', array(
                'label' => 'Possible file deletion path traversal #21412',
                'description' => 'Fix path traversal in emotion import',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_recover_password_21640', array(
                'label' => 'Fix recovering password with wrong mail #21640',
                'description' => 'Fix recovering password for mails with unicode characters',
                'value' => 1,
            ));
        }

        // Fix plugin manager integrity
        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.4.4', '<')
        ) {
            $form->setElement('checkbox', 'fix_plugin_manager_integrity_21776', array(
                'label' => 'Improve plugin manager integrity #21776',
                'description' => 'Add additional checks to improve plugin manager integrity',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.5.1', '<')
        ) {
            $form->setElement('checkbox', 'fix_xss_in_newsletter_22065', array(
                'label' => 'Fix XSS in newsletter #22065',
                'description' => 'Adds escaping in newsletter email field',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.5.1', '<')
        ) {
            $form->setElement('checkbox', 'fix_phar_deserilization_22386', array(
                'label' => 'Phar deserialization in REST API #22386',
                'description' => 'Adds phar check to the REST API',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.5.4', '<')
        ) {
            $form->setElement('checkbox', 'fix_authenticated_rce_product_export', array(
                'label' => 'Fix possible authenticated remote code execution in product exports #23009',
                'description' => 'Adds file name checks to product export file names',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.5.4', '<')
        ) {
            $form->setElement('checkbox', 'fix_esd_upload_blacklist_23010', array(
                'label' => 'Blacklist for ESD uploads #23010',
                'description' => 'Adds a file extension blacklist for ESD uploads',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.3.0', '>=')
            && version_compare($version, '5.5.4', '<')
        ) {
            $form->setElement('checkbox', 'fix_path_traversal_media_fallback', array(
                'label' => 'Fix possible path traversal on media live migration #23011',
                'description' => 'Adds additional checks to paths on live media migration',
                'value' => 1,
            ));
        }

        if (version_compare($version, '5.5.4', '<')) {
            $form->setElement('checkbox', 'fix_validation_bypass_23012', array(
                'label' => 'Fix validation bypass #23012',
                'description' => 'Overwrites params in the HttpRequest class to prevent a validation bypass',
                'value' => 1,
            ));
        }

        if (version_compare($version, '5.5.7', '<')) {
            $form->setElement('checkbox', 'fix_perform_order_23016', array(
                'label' => 'Fix perform order bypass #23016',
                'description' => 'Checks for invalid parameters, to prevent a validation bypass',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_xss_23319', array(
                'label' => 'Fix possible XSS on forms #23319',
                'description' => 'Adds validation and escaping on a parameter of the Forms controller',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.5.7', '<')
        ) {
            $form->setElement('checkbox', 'fix_session_fixation_23428', array(
                'label' => 'Fix possible session fixation #23428',
                'description' => 'Removes possibility to create links with specific session ids',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.2.0', '>=')
            && version_compare($version, '5.5.7', '<')
        ) {
            $form->setElement('checkbox', 'fix_xsrf_token_leakage_23007', array(
                'label' => 'Fix XSRF TokenLeakage #23007',
                'description' => 'Makes sure the CSRF token is only being send to this server',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.5.8', '<')
        ) {
            $form->setElement('checkbox', 'fix_dqli_in_api_23626', array(
                'label' => 'Fix possible authenticated dql injection #23626',
                'description' => 'Fixes a possible dql injection in API and backend controllers',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_sqli_in_checkesd_23766', array(
                'label' => 'Fix possible sql injection #23766',
                'description' => 'Casts parameters of function sArticles::sCheckIfEsd to prevent a possible sql injection',
                'value' => 1,
            ));

            $form->setElement('checkbox', 'fix_xss_in_exception_23603', array(
                'label' => 'Fix possible non-persistent XSS #23603',
                'description' => 'Fixes a possible XSS on errors pages',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.0.0', '>=')
            && version_compare($version, '5.7.18', '<')
        ) {
            $form->setElement('checkbox', 'decorate_email_validator', array(
                'label' => 'Use PHP library e-mail validation',
                'description' => 'Use the PHP library validation instead custom e-mail validation.',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.1.0', '>=')
            && version_compare($version, '5.5.9', '<')
        ) {
            $form->setElement('checkbox', 'fix_xxe_in_product_stream_24068', array(
                'label' => 'Fix possible authenticated remote code execution #24068',
                'description' => 'Fixes a possible authenticated remote code execution in backend.',
                'value' => 1,
            ));
        }

        if ($version === '5.6.0') {
            $form->setElement('checkbox', 'fix_xss_24473', array(
                'label' => 'Fix possible non-persistent XSS #24473',
                'description' => 'Fixes a possible XSS on storefront',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.4.5', '>=')
            && version_compare($version, '5.6.2', '<')
        ) {
            $form->setElement('checkbox', 'fix_invalid_doi_user_24590', array(
                'label' => 'Fix possible invalid DOI authentication #24590',
                'description' => 'Fixes a timing issue with the DOI registration, where a user can be logged in as another user. Only necessary when using the double opt in registration.',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.6.7', '<')
        ) {
            $form->setElement('checkbox', 'fix_cached_blog_comment_25409', array(
                'label' => 'Fix blog comment form content being cached #25409',
                'description' => 'Prevents the content of the blog comment form from being cached.',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.3.0', '>=')
            && (version_compare($version, '5.6.9', '<') || (version_compare($version, '5.6.10', '>') && version_compare($version, '5.7.14', '<')))
        ) {
            $form->setElement('checkbox', 'xss_backend_customer_mail_25773', array(
                'label' => 'Fixes customer mail escape #25773',
                'description' => 'Escapes customer mail in customer module',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.6.9', '<')
            || (version_compare($version, '5.6.10', '>') && version_compare($version, '5.7.14', '<'))
        ) {
            $form->setElement('checkbox', 'xss_backend_newsletter_25772', array(
                'label' => 'Fixes XSS in newsletter module #25772',
                'description' => 'Prevents Javascript execution in Newsletter Manager',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.6.9', '<')
            || (version_compare($version, '5.6.10', '>') && version_compare($version, '5.7.14', '<'))
        ) {
            $form->setElement('checkbox', 'xss_backend_emotion_elements_25771', array(
                'label' => 'Fixes possible XSS for emotion elements. #25771',
                'description' => 'Escapes values in emotion elements to prevent an XSS injection.',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.6.10', '<')
        ) {
            $form->setElement('checkbox', 'json_response_in_error_controller_26001', array(
                'label' => 'Prevent the Error controller from returning JSON-encoded data. #26001',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.6.10', '<')
            || (version_compare($version, '5.6.10', '>') && version_compare($version, '5.7.14', '<'))
        ) {
            $form->setElement('checkbox', 'xss_backend_product_name_26050', array(
                'label' => 'Escape product names properly. #26050',
                'value' => 1,
            ));
        }

        if (version_compare($version, '5.7.7', '<')) {
            $form->setElement('checkbox', 'arbitrary_redirect_26435', array(
                'label' => 'Prevent arbitrary redirect via shopware.php file. #26435',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.7.3', '>=')
            && version_compare($version, '5.7.7', '<')
        ) {
            $form->setElement('checkbox', 'session_invalidation_upon_password_change_26448', array(
                'label' => 'Invalidate all sessions for a given customer account when the password is changed. #26448',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.7.9', '<')
        ) {
            $form->setElement('checkbox', 'fix_array_xss_26657', array(
                'label' => 'Fixed possible XSS from arrays as request parameters. #26657',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.7.12', '<')
        ) {
            $form->setElement('checkbox', 'fix_xss_26748', array(
                'label' => 'Fixed possible decoded XSS. #26748',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.7.9', '<')
        ) {
            $form->setElement('checkbox', 'fix_csrf_token_26662', array(
                'label' => 'Create new csrf token with login and logout. #26662',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.2.21', '>=')
            && version_compare($version, '5.7.9', '<')
        ) {
            $form->setElement('checkbox', 'password_reset_token_invalidation_26666', array(
                'label' => 'Invalidate all existing password reset tokens prior to creating a new one. #26666',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.7.15', '<')
        ) {
            $form->setElement('checkbox', 'fixAclResource_26913', array(
                'label' => 'Checks also backend acl resources with underscore. #26913',
                'value' => 1,
            ));
        }

        if (
            version_compare($version, '5.7.15', '<')
        ) {
            $form->setElement('checkbox', 'fixSensitiveData_26909', array(
                'label' => 'Removes sensitiv data from the customer module. #26909',
                'value' => 1,
            ));
        }

        return array(
            'success' => true,
            'invalidateCache' => array('proxy', 'frontend', 'backend', 'theme'),
        );
    }

    /**
     * @return void
     */
    public function fixArbitraryRedirect_26435(Enlight_Controller_EventArgs $args)
    {
        $version = $this->swVersion();
        if (version_compare($version, '5.7.7', '>=') || !$this->Config()->get('arbitrary_redirect_26435')) {
            return;
        }

        if (!Shopware()->Container()->initialized('shop')) {
            return;
        }

        $request = $args->getRequest();

        if (strpos($request->getPathInfo(), '/shopware.php/') === false) {
            return;
        }

        if (!$this->get('config')->get('preferBasePath')) {
            return;
        }

        $cleanedUpRequestUri = preg_replace('/\/{2,}/', '/', $request->getRequestUri());
        if ($request->getRequestUri() !== $cleanedUpRequestUri) {
            $request->setRequestUri($cleanedUpRequestUri);
        }
    }

    public function decorateEmailValidation()
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.7.18', '>=')
        ) {
            return;
        }

        if (!$this->Config()->get('decorate_email_validator')) {
            return;
        }

        $originalService = Shopware()->Container()->get('validator.email');

        Shopware()->Container()->set(
            'validator.email',
            new EmailValidatorDecorator($originalService)
        );
    }

    public function fixOperatorsForDql23626(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.8', '>=')
            || class_exists('\Shopware\Components\Model\QueryOperatorValidator')
            || !$this->Config()->get('fix_dqli_in_api_23626')
        ) {
            return;
        }

        require_once __DIR__ . '/Components/Model/QueryOperatorValidator.php';
        require_once __DIR__ . '/Components/Model/QueryBuilder.php';
        require_once __DIR__ . '/Components/Model/ModelManager.php';

        /** @var \Shopware\Components\DependencyInjection\Container $container */
        $container = $args->get('subject');

        $orgModelManager = $container->get('models');

        $modelManager = \SwagSecurity\Components\Model\ModelManager::createInstance(
            $orgModelManager->getConnection(),
            $orgModelManager->getConfiguration(),
            $orgModelManager->getEventManager(),
            new \Shopware\Components\Model\QueryOperatorValidator()
        );

        $container->set('models', $modelManager);

        $args->setReturn($modelManager);
    }

    public function castParametersOnCheckEsd23766(Enlight_Hook_HookArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.8', '>=')
            || !$this->Config()->get('fix_sqli_in_checkesd_23766')
        ) {
            return;
        }

        $args->set('id', (int) $args->get('id'));
        $args->set('detailsID', (int) $args->get('detailsID'));
    }

    public function fixExceptionMessage23603(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.8', '>=')
            || !$this->Config()->get('fix_xss_in_exception_23603')
        ) {
            return;
        }

        /** @var \Shopware_Controllers_Frontend_Error $subject */
        $subject = $args->getSubject();
        $request = $subject->Request();
        $action = strtolower($request->getActionName());

        if ($action !== 'genericerror') {
            return;
        }

        $subject->View()->addTemplateDir(__DIR__ . '/Views/');

        if ($request->getModuleName() === 'backend') {
            $assigns = $subject->View()->getAssign();
            $subject->View()->loadTemplate('backend/exception.tpl');
            $subject->View()->assign($assigns);
        } else {
            $subject->View()->extendsTemplate('frontend/exception.tpl');
        }
    }

    public function fixXSSNewsletter22065ExtendXssRegex(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.1', '>=')
            || !$this->Config()->get('fix_xss_in_newsletter_22065')
        ) {
            return;
        }

        Shopware()->Plugins()->Frontend()->InputFilter()->xssRegex .= '|\bdata-\w+';
    }

    public function fixXSSNewsletter22065ExtendTemplate(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.1', '>=')
            || !$this->Config()->get('fix_xss_in_newsletter_22065')
        ) {
            return;
        }

        $args->getSubject()->View()->addTemplateDir(__DIR__ . '/Views/');
    }

    public function getPluginMetaData21776(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.4.4', '>=')
            || !$this->Config()->get('fix_plugin_manager_integrity_21776')
        ) {
            return;
        }
        /** @var \Shopware_Controllers_Backend_PluginManager $subject */
        $subject = $args->getSubject();
        $request = $subject->Request();

        if (strtolower($request->getActionName()) !== 'metadownload') {
            return;
        }

        /** @var \Shopware\Bundle\PluginInstallerBundle\Struct\MetaStruct $data */
        $data = $subject->View()->getAssign('data');
        $technicalName = $request->getParam('technicalName');

        if (empty($data) || empty($technicalName)) {
            return;
        }

        $backendSession = Shopware()->Container()->get('BackendSession');

        $backendSession->offsetSet('plugin_manager_meta_download', array(
            'metaStruct' => $data,
            'technicalName' => $technicalName,
        ));
    }

    public function fixMetaDataIntegrity21776(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.4.4', '>=')
            || !$this->Config()->get('fix_plugin_manager_integrity_21776')
        ) {
            return;
        }

        /** @var \Shopware_Controllers_Backend_PluginManager $subject */
        $subject = $args->getSubject();
        $request = $subject->Request();
        $action = strtolower($request->getActionName());

        if ($action !== 'rangedownload' && $action !== 'extract') {
            return;
        }

        $backendSession = Shopware()->Container()->get('BackendSession');
        $data = $backendSession->offsetGet('plugin_manager_meta_download');

        if (empty($data)) {
            throw new Exception(sprintf('This request (%s) seems to be malicious and has been denied. This is enforced by the fix 21776 from the SwagSecurityPlugin', $action));
        }

        /** @var \Shopware\Bundle\PluginInstallerBundle\Struct\MetaStruct $meta */
        $meta = $data['metaStruct'];
        $technicalName = $data['technicalName'];

        switch ($action) {
            case 'rangedownload':
                $request->setParams(array(
                    'uri' => $meta->getUri(),
                    'size' => $meta->getSize(),
                    'sha1' => $meta->getSha1(),
                    'fileName' => $meta->getFileName(),
                ));
                break;
            case 'extract':
                $downloadsDir = Shopware()->Container()->getParameter('kernel.root_dir') . '/files/downloads/';
                $filePath = $downloadsDir . $meta->getFileName();
                $request->setParams(array(
                    'technicalName' => $technicalName,
                    'fileName' => $filePath,
                ));
                break;
        }
    }

    public function afterInit()
    {
        parent::afterInit();
        $this->registerNamespaces();
    }

    public function registerNamespaces()
    {
        $this->Application()->Loader()->registerNamespace(
            'Shopware\Plugins\SwagSecurity',
            $this->Path()
        );
    }

    /**
     * @throws Enlight_Exception
     */
    public function fixRecoverPassword21640(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (version_compare($version, '5.4.3', '>=') || !$this->Config()->get('fix_recover_password_21640')) {
            return;
        }

        /** @var \Shopware_Controllers_Frontend_Account $controller */
        $controller = $args->getSubject();

        $request = $controller->Request();

        if ($controller->Request()->getActionName() !== 'password' || !$request->isPost()) {
            return;
        }

        $submittedMail = $request->getParam('email');

        $userId = $this->get('modules')->Admin()->sGetUserByMail($submittedMail);

        if (!$userId) {
            return;
        }

        $mail = $this->getUserMailById((int) $userId);
        $request->setParam('email', $mail);
    }

    /**
     * Delete deprecated events
     */
    public function deleteDeprecatedEvents()
    {
        $events = array(
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixXXE_19896',
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixXSS_19834',
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixSQLi_19895',
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixCSRFFinishCheckoutSaveOrder20983',
        );

        foreach ($events as $deprecatedEvent) {
            $this->deleteEvent($deprecatedEvent);
        }
    }

    /**
     * Fix path traversal media upload
     */
    public function fixPathTraversal21151(Enlight_Controller_ActionEventArgs $args)
    {
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $fix = (version_compare($version, '5.4.3', '<') && $this->Config()->get('fix_path_traversal_21151'));
        $action = strtolower($request->getActionName());
        $protectedActions = array('put', 'post');

        if (!$fix || !\in_array($action, $protectedActions, true)) {
            return;
        }

        $postParams = $request->getPost();

        if (\array_key_exists('name', $postParams)) {
            $postParams['name'] = basename($postParams['name']);
        }

        $request->setPost($postParams);
    }

    /**
     * Fix path traversal on emotion import
     */
    public function fixPathTraversal21412(Enlight_Controller_ActionEventArgs $args)
    {
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $fix = (version_compare($version, '5.4.3', '<') && $this->Config()->get('fix_path_traversal_21412'));
        $action = strtolower($request->getActionName());
        $protectedActions = array('import', 'afterimport');
        $filePath = basename(trim($request->getParam('filePath')));

        if (!$fix || !$filePath || !\in_array($action, $protectedActions, true)) {
            return;
        }

        $path = sprintf(
            '%s%s%s%s%s',
            Shopware()->Container()->getParameter('kernel.root_dir'),
            \DIRECTORY_SEPARATOR,
            implode(\DIRECTORY_SEPARATOR, array('files', 'downloads')),
            \DIRECTORY_SEPARATOR,
            $filePath
        );

        $request->setParam('filePath', $path);
    }

    /**
     * @throws Exception
     */
    public function fixSQLSecurity_21404(Enlight_Controller_ActionEventArgs $args)
    {
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $action = $request->getActionName();

        $fix = (
            version_compare($version, '5.1.0', '>=')
            && version_compare($version, '5.4.3', '<')
            && $this->Config()->get('fix_sql_injection_21404')
        );

        if (!$fix || !\in_array($case = strtolower($action), array('create', 'update', 'delete'))) {
            return;
        }

        $tableName = $request->getParam('tableName');
        $columnName = $request->getParam('columnName');

        $this->validateField($tableName);
        $this->validateField($columnName);

        $request->setParam('tableName', $tableName);
        $request->setParam('columnName', $columnName);

        if ($case === 'update') {
            $originalName = $request->getParam('originalName');
            $defaultValue = $this->filterDefaultValue($request->getParam('defaultValue'));

            $this->validateField($originalName);

            $request->setParam('originalName', $originalName);
            $request->setParam('defaultValue', $defaultValue);
        }
    }

    /**
     * @throws Exception
     */
    public function fixCSRFAddArticle20983(Enlight_Controller_ActionEventArgs $args)
    {
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $action = $request->getActionName();

        $fix = (version_compare($version, '5.4.0', '<') && $this->Config()->get('fix_csrf_20938'));

        if (!$fix) {
            return;
        }

        switch (strtolower($action)) {
            case 'ajaxaddarticlecart':
            case 'ajax_add_article_cart':
                if ($request->isXmlHttpRequest()) {
                    return;
                }
                break;
            case 'addarticle':
            case 'add_article':
                if (!$request->isGet() || $request->isXmlHttpRequest()) {
                    return;
                }
                break;
            default:
                return;
        }

        throw new Exception(sprintf('This request (%s) seems to be malicious and has been denied. This is enforced by the fix 20938 from the SwagSecurityPlugin', $action));
    }

    /**
     * @throws Exception
     */
    public function fixCSRFFinishCheckout20983(Enlight_Controller_ActionEventArgs $args)
    {
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $action = $request->getActionName();

        $fix = (version_compare($version, '5.4.0', '<') && $this->Config()->get('fix_csrf_20938'));

        if (!$fix || strtolower($action) !== 'finish') {
            return;
        }

        $this->finishActionCalled = true;
    }

    /**
     * Fixes both 20983 and 21593
     */
    public function fixCSRFFinishCheckoutSaveOrder20983(Enlight_Hook_HookArgs $args)
    {
        $this->deleteDeprecatedEvents();
    }

    /**
     * @throws SwagSecurityCSRFCheckoutException
     */
    public function fixCSRFFinishCheckoutSaveOrderBefore20983(Enlight_Hook_HookArgs $args)
    {
        $version = $this->swVersion();

        // Fixes #21593
        if (version_compare($version, '5.4.3', '<') && $this->Config()->get('fix_currency_change_21593')) {
            $basket = $args->getSubject()->View()->getAssign('sBasket');

            // in case a plugin uses the save order method without invoking the getBasket method (e.g. SwagAboCommerce)
            if (!\array_key_exists('sCurrencyId', $basket)) {
                $basket = array_merge($basket, $this->getCurrencyData());
            }
            $this->updateCurrencyDependencies((int) $basket['sCurrencyId']);
        }

        $fix = (version_compare($version, '5.4.0', '<') && $this->Config()->get('fix_csrf_20938'));

        if (!$fix) {
            return;
        }

        if (!$this->finishActionCalled) {
            throw new SwagSecurityCSRFCheckoutException('This request seems to be malicious and has been denied. This is enforced by the fix 20989 from the SwagSecurityPlugin');
        }

        /** @var Enlight_Controller_Request_Request $request */
        $request = $args->getSubject()->Request();

        if (!$request->isGet()) {
            return;
        }
        throw new SwagSecurityCSRFCheckoutException('This request seems to be malicious and has been denied. This is enforced by the fix 20989 from the SwagSecurityPlugin');
    }

    /**
     * @throws Enlight_Exception
     */
    public function fixCSRFFinishCheckoutPostDispatch20983(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        $fix = (version_compare($version, '5.4.0', '<') && $this->Config()->get('fix_csrf_20938'));

        if (!$fix) {
            return;
        }

        /** @var Enlight_Controller_Front $controller */
        $controller = $args->getSubject();

        /** @var Enlight_Controller_Request_RequestHttp $request */
        $request = $controller->Request();
        if (strtolower($request->getControllerName()) !== 'checkout') {
            return;
        }

        /** @var Enlight_Controller_Response_ResponseHttp $response */
        $response = $controller->Response();

        if (!$response->getException()) {
            return;
        }

        $exceptions = $response->getException();
        foreach ($exceptions as $exception) {
            if ($exception instanceof SwagSecurityCSRFCheckoutException) {
                $url = $controller->Router()->assemble(array(
                    'module' => 'frontend',
                    'controller' => 'checkout',
                    'action' => 'confirm',
                ));
                $response->setRedirect($url);
            }
        }
    }

    public function fixXSSNewsletter20878(Enlight_Controller_ActionEventArgs $args)
    {
        $major = (int) $this->swVersion('major');
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $action = $request->getActionName();

        $fix = (
            $major === 5
            && version_compare($version, '5.3.0', '>=')
            && version_compare($version, '5.3.7', '<')
            && $this->Config()->get('fix_xss_20878')
        );

        if (!$fix || strtolower($action) !== 'detail') {
            return;
        }

        $page = $request->getParam('p', 1);
        $request->setParam('p', (int) $page);
    }

    public function fixXSSAddress20878(Enlight_Event_EventArgs $args)
    {
        $major = (int) $this->swVersion('major');
        $version = $this->swVersion();
        $request = $args->getSubject()->Request();
        $action = $request->getActionName();

        $fix = (
            $major === 5
            && version_compare($version, '5.2.0', '>=')
            && version_compare($version, '5.3.7', '<')
            && $this->Config()->get('fix_xss_20878')
        );

        if (!$fix || strtolower($action) !== 'ajaxselection') {
            return;
        }

        $extraData = $request->getParam('extraData', array());

        /** @var string $data */
        $extraData = array_map(function ($data) {
            // only allow alphanumeric characters, commas and spaces
            return preg_replace('/[^A-Za-z0-9 ,]/', '', $data);
        }, $extraData);

        $request->setParam('extraData', $extraData);
    }

    /**
     * Old XXE fix method
     */
    public function fixXXE_19896(Enlight_Event_EventArgs $args)
    {
        $this->deleteEvent(
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixXXE_19896'
        );
    }

    /**
     * @throws \InvalidArgumentException
     */
    public function new_fixXXE_19896(Enlight_Event_EventArgs $args)
    {
        $major = (int) $this->swVersion('major');
        $version = $this->swVersion();

        $fix = ($major === 5 && version_compare($version, '5.3.4', '<')) && $this->Config()->get('fix_xxe_19896');

        if (!$fix) {
            return;
        }

        $request = $args->getSubject()->Request();
        $actionName = strtolower(trim($request->getActionName()));

        if ($actionName !== 'loadpreview') {
            return;
        }

        $sort = $request->getParam('sort');

        if (empty($sort) || (\is_string($sort) && strtolower(trim($sort)) === 'null')) {
            return;
        }

        if (!\is_array($sort)) {
            throw new \InvalidArgumentException('Sort parameter is expected to be an array or a literal null string', 1);
        }

        foreach ($sort as $key => $value) {
            $classToLower = strtolower(trim($key, '\\'));
            $isShopwareClass = (strpos($classToLower, 'shopware') === 0);

            if (!$isShopwareClass) {
                throw new \InvalidArgumentException("Invalid sorting specified \"$key\"", 2);
            }

            if (isset($value['direction'])) {
                if (!\is_string($value['direction']) || !\in_array(strtolower($value['direction']), array('desc', 'asc'), true)) {
                    throw new \InvalidArgumentException('Invalid sorting specified', 3);
                }
            }
        }
    }

    public function fix_xxe_in_product_stream_24068(Enlight_Event_EventArgs $args)
    {
        $major = (int) $this->swVersion('major');
        $version = $this->swVersion();

        $fix = ($major === 5 && version_compare($version, '5.5.9', '<')) && $this->Config()->get('fix_xxe_in_product_stream_24068');

        if (!$fix) {
            return;
        }

        $request = $args->getSubject()->Request();
        $actionName = strtolower(trim($request->getActionName()));

        if ($actionName !== 'loadpreview') {
            return;
        }

        $sort = $request->getParam('sort');

        if (empty($sort) || (\is_string($sort) && strtolower(trim($sort)) === 'null')) {
            return;
        }

        if (!\is_array($sort)) {
            throw new \InvalidArgumentException('Sort parameter is expected to be an array or a literal null string', 1);
        }

        foreach ($sort as $key => $value) {
            if (!\array_key_exists('Shopware\Bundle\SearchBundle\SortingInterface', class_implements($key))) {
                throw new \InvalidArgumentException(sprintf('Class %s has to implement the interface %s', $key, 'Shopware\Bundle\SearchBundle\SortingInterface'));
            }
        }
    }

    /**
     * Old XSS fix event
     */
    public function fixXSS_19834(Enlight_Event_EventArgs $args)
    {
        $this->deleteEvent(
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixXSS_19834'
        );
    }

    /**
     * @throws \Exception
     */
    public function new_fixXSS_19834(Enlight_Event_EventArgs $args)
    {
        $major = (int) $this->swVersion('major');
        $version = $this->swVersion();
        $fix = ($major === 5 && version_compare($version, '5.3.4', '<')) && $this->Config()->get('fix_xss_19834');

        if (!$fix) {
            return;
        }

        $request = $args->getSubject()->Request();
        $action = $request->getActionName();

        $protectedActions = array('save', 'update', 'create');

        if (!\in_array(strtolower($action), $protectedActions, true)) {
            return;
        }

        $this->cleanValues($request);
    }

    /**
     * Old SQLi Fix method
     */
    public function fixSQLi_19895(Enlight_Event_EventArgs $args)
    {
        $this->deleteEvent(
            'Shopware_Plugins_Core_SwagSecurity_Bootstrap::fixSQLi_19895'
        );
    }

    /**
     * @throws \InvalidArgumentException
     */
    public function new_fixSQLi_19895(Enlight_Event_EventArgs $args)
    {
        $major = (int) $this->swVersion('major');
        $version = $this->swVersion();

        $fix = ($major === 5 && version_compare($version, '5.3.4', '<')) && $this->Config()->get('fix_sql_injection_19895');

        $actionName = strtolower(trim($args->getSubject()->Request()->getActionName()));

        if (!$fix || $actionName !== 'loadpreview') {
            return;
        }

        $validOperators = array(
            'OPERATOR_EQ' => '=',
            'OPERATOR_NEQ' => '!=',
            'OPERATOR_LT' => '<',
            'OPERATOR_LTE' => '<=',
            'OPERATOR_GT' => '>',
            'OPERATOR_GTE' => '>=',
            'OPERATOR_NOT_IN' => 'NOT IN',
            'OPERATOR_IN' => 'IN',
            'OPERATOR_BETWEEN' => 'BETWEEN',
            'OPERATOR_STARTS_WITH' => 'STARTS_WITH',
            'OPERATOR_ENDS_WITH' => 'ENDS_WITH',
            'OPERATOR_CONTAINS' => 'CONTAINS',
        );

        $subject = $args->getSubject();
        $request = $subject->Request();
        $start = $request->getParam('start');
        $limit = $request->getParam('limit');
        $conditions = $request->getParam('conditions');

        if ($start !== null) {
            $start = (int) $start;
            if ($start < 0) {
                throw new \InvalidArgumentException('Invalid start offset', 1);
            }
            $request->setParam('start', $start);
        }

        if ($limit !== null) {
            $limit = (int) $limit;
            if ($limit < 0) {
                throw new \InvalidArgumentException('Invalid limit offset', 2);
            }
            $request->setParam('limit', $limit);
        }

        if (empty($conditions)) {
            return;
        }

        $decoded = json_decode($conditions, true);

        if (json_last_error() !== \JSON_ERROR_NONE) {
            throw new \InvalidArgumentException('Invalid JSON', 3);
        }

        $isValidFieldRegexp = '/^[a-zA-Z_][a-zA-Z0-9_]*$/';

        foreach ($decoded as &$condition) {
            if (isset($condition['field'])) {
                if (!\is_string($condition['field'])) {
                    throw new \InvalidArgumentException('String value expected for field name', 4);
                }

                $isValidFieldName = preg_match($isValidFieldRegexp, $condition['field']);

                if (!$isValidFieldName) {
                    throw new \InvalidArgumentException('Invalid field name', 5);
                }
            }

            if (isset($condition['operator'])) {
                if (!\is_string($condition['operator'])) {
                    throw new \InvalidArgumentException('String value expected for operator', 6);
                }

                if (!\in_array(strtoupper($condition['operator']), $validOperators, true)) {
                    throw new \InvalidArgumentException(
                        sprintf('Must specify an operator, please use one of: %s', implode(', ', $validOperators)),
                        7
                    );
                }
            }
        }

        $request->setParam('conditions', json_encode($decoded));
    }

    public function fixVoucherIdParameter(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }

        $args->set('voucherId', (int) $args->get('voucherId'));
    }

    public function fixOrderIdParameter(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $args->set('orderId', (int) $args->get('orderId'));
    }

    public function fixMediaIdParameter(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $args->set('albumId', (int) $args->get('albumId'));
    }

    public function fixUsedIdsFilter(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $filters = $args->get('filter');
        if (!\is_array($filters)) {
            return;
        }

        $new = array();
        foreach ($filters as $key => $filter) {
            if ($filter['property'] === 'usedIds' || $filter['property'] === 'onlyIds') {
                $filter['value'] = $this->castIntArray($filter['value']);
            }
            $new[$key] = $filter;
        }
        $args->set('filter', $new);
    }

    public function fixDispatchIdParameter(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $args->set('dispatchId', (int) $args->get('dispatchId'));
    }

    public function fixCustomerIdParameter(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $args->set('customerId', (int) $args->get('customerId'));
    }

    public function getBackendListQueryBuilderBlogRepository(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $ids = $args->get('blogCategoryIds');
        $ids = $this->castIntArray($ids);
        $args->set('blogCategoryIds', $ids);
    }

    public function getFilterQueryBuilderBlogRepository(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $ids = $args->get('categoryIds');
        $ids = $this->castIntArray($ids);
        $args->set('categoryIds', $ids);
    }

    public function fixBlogRepositoryListQuery(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $ids = $args->get('blogCategoryIds');
        $ids = $this->castIntArray($ids);
        $args->set('blogCategoryIds', $ids);
    }

    public function fixIdsArray(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $ids = $args->get('ids');
        $ids = $this->castIntArray($ids);
        $args->set('ids', $ids);
    }

    public function getOrderDocumentsQuery(Enlight_Hook_HookArgs $args)
    {
        if (!$this->Config()->get('fix_sql_injection')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        $ids = $args->get('orderIds');
        $ids = $this->castIntArray($ids);
        $args->set('orderIds', $ids);
    }

    public function fixWidgetsSecurity(Enlight_Event_EventArgs $args)
    {
        /** @var \Shopware_Controllers_Backend_Widgets $subject */
        $subject = $args->getSubject();
        $request = $subject->Request();

        if (strtolower($request->getActionName()) !== 'sendmailtomerchant') {
            return;
        }
        if (!$this->Config()->get('fix_widgets_smarty_security')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.26', '>=')) {
            return;
        }
        include __DIR__ . '/Swag_Smarty_Security.php';

        $subject->View()->Engine()->enableSecurity(
            new Swag_Smarty_Security(
                $subject->View()->Engine(),
                $this->getSmartySecurityConfig()
            )
        );
    }

    public function fixMailSecurity(Enlight_Event_EventArgs $args)
    {
        /** @var \Shopware_Controllers_Backend_Mail $subject */
        $subject = $args->getSubject();
        $request = $subject->Request();

        if (strtolower($request->getActionName()) !== 'verifysmarty') {
            return;
        }
        if (!$this->Config()->get('fix_mail_smarty_security')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.26', '>=')) {
            return;
        }
        include __DIR__ . '/Swag_Smarty_Security.php';

        $subject->View()->Engine()->enableSecurity(
            new Swag_Smarty_Security(
                $subject->View()->Engine(),
                $this->getSmartySecurityConfig()
            )
        );
    }

    public function fixExportOutput(Enlight_Event_EventArgs $args)
    {
        if (!$this->Config()->get('fix_export_output')) {
            return;
        }
        if (version_compare($this->swVersion(), '5.2.25', '>=')) {
            return;
        }
        /** @var Enlight_Controller_Action $controller */
        $controller = $args->getSubject();

        $request = $controller->Request();

        $format = $request->getParam('format', null);
        if ($format !== null) {
            $request->setParam('format', htmlspecialchars($format));
        }

        $type = $request->getParam('type', null);
        if ($type !== null) {
            $request->setParam('type', htmlspecialchars($type));
        }
    }

    public function fixUnallowedCurrencyChangeGetBasket21593(Enlight_Hook_HookArgs $args)
    {
        $version = $this->swVersion();

        if (version_compare($version, '5.4.3', '>=') || !$this->Config()->get('fix_currency_change_21593')) {
            return;
        }

        $basket = $args->getReturn();
        $basket = array_merge($basket, $this->getCurrencyData());
        $args->setReturn($basket);
    }

    public function fixUnallowedCurrencyChangeFilterParams21593(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();

        if (version_compare($version, '5.4.3', '>=') || !$this->Config()->get('fix_currency_change_21593')) {
            return;
        }

        $orderParams = $args->getReturn();
        $sOrder = $args->getSubject();

        $basket = $sOrder->sBasketData;

        if (\array_key_exists('sCurrencyName', $basket) && \array_key_exists('sCurrencyFactor', $basket)) {
            $currencyName = $basket['sCurrencyName'];
            $currencyFactor = $basket['sCurrencyFactor'];
        } else {
            $currencyData = $this->getCurrencyData();
            $currencyName = $currencyData['sCurrencyName'];
            $currencyFactor = $currencyData['sCurrencyFactor'];
        }

        $orderParams['currency'] = $currencyName;
        $orderParams['currencyFactor'] = $currencyFactor;

        $args->setReturn($orderParams);
    }

    public function fixPharDeserialization22386(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();

        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.1', '>=')
            || !$this->Config()->get('fix_phar_deserilization_22386')
        ) {
            return;
        }

        $allowedProtocols = array(
            'ftp',
            'ftps',
            'http',
            'https',
            'file',
            'data',
        );

        /** @var Enlight_Controller_Action $controller */
        $controller = $args->get('subject');

        $action = strtolower($controller->Request()->getActionName());

        if (\in_array($action, array('batch', 'put'))) {
            $fileData = parse_url(html_entity_decode($controller->Request()->getPost('file')));
            $scheme = $fileData['scheme'];

            if (!\in_array($scheme, $allowedProtocols, true)) {
                throw new RuntimeException('Invalid URL scheme found!');
            }
        }
    }

    public function fixEsdUploadSecurity23010(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        /** @var Enlight_Controller_Action $controller */
        $controller = $args->get('subject');
        $action = strtolower($controller->Request()->getActionName());
        $view = $controller->View();

        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.4', '>=')
            || !$this->Config()->get('fix_esd_upload_blacklist_23010')
            || !\in_array($action, array('uploadesdfile'))
        ) {
            return;
        }

        $blacklist = array(
            'php',
            'php3',
            'php4',
            'php5',
            'phtml',
            'cgi',
            'pl',
            'sh',
            'com',
            'bat',
            '',
            'py',
            'rb',
        );

        $fileBag = new \Symfony\Component\HttpFoundation\FileBag($_FILES);
        /** @var Symfony\Component\HttpFoundation\File\UploadedFile $file */
        $file = $fileBag->get('fileId');

        if ($file === null) {
            $view->assign(array('success' => false));

            return;
        }

        $extension = strtolower($file->getClientOriginalExtension());
        $name = strtolower($file->getClientOriginalName());

        if ($name[0] !== '.' && !\in_array($extension, $blacklist, true)) {
            return;
        }

        if (
            version_compare($version, '5.3.0', '>=')
        ) {
            $e = new \Shopware\Bundle\MediaBundle\Exception\MediaFileExtensionIsBlacklistedException($extension);
            exit(json_encode(array(
                'success' => false,
                'message' => $e->getMessage(),
                'exception' => array(
                    '_class' => \get_class($e),
                    'extension' => $extension,
                ),
            )));
        }

        $auth = Shopware()->Auth();
        $user = $auth->getIdentity();

        if ($name[0] === '.') {
            $msg = (stripos($user->locale->getLocale(), 'de') === 0) ? 'Der Dateiname darf nicht mit Punkt beginnen.' : 'The file name must not begin with a dot.';
        } else {
            $msg = (stripos($user->locale->getLocale(), 'de') === 0) ? 'Die Dateiendung "%s" ist nicht erlaubt.' : 'The file extension "%s" is blacklisted.';
        }

        exit(json_encode(array(
            'success' => false,
            'error' => sprintf($msg, $extension),
        )));
    }

    public function fixPathTraversal23011(Enlight_Event_EventArgs $args)
    {
        /** @var \Shopware_Controllers_Frontend_Media $subject */
        $subject = $args->getSubject();

        $version = $this->swVersion();

        if (
            version_compare($version, '5.3.0', '<')
            || version_compare($version, '5.5.4', '>=')
            || !$this->Config()->get('fix_path_traversal_media_fallback')
        ) {
            return;
        }

        /** @var \Shopware\Bundle\MediaBundle\MediaService $mediaService */
        $mediaService = $this->get('shopware_media.media_service');

        if ($mediaService->getAdapterType() !== 'local') {
            return;
        }

        if (!Shopware()->Container()->hasParameter('shopware.cdn.liveMigration') || !Shopware()->Container()->getParameter('shopware.cdn.liveMigration')) {
            return;
        }

        $requestedImage = $subject->Request()->getRequestUri();
        $path = \League\Flysystem\Util::normalizePath($requestedImage);

        if (strpos($path, 'media/') !== 0) {
            throw new RuntimeException('Path Traversal detected');
        }
    }

    /**
     * @throws Exception
     */
    public function fixValidationBypass23012(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();

        if (
            version_compare($version, '5.5.4', '>=')
            || !$this->Config()->get('fix_validation_bypass_23012')
        ) {
            return;
        }

        /** @var Enlight_Controller_Request_Request $request */
        $request = $args->getSubject()->Request();

        $moduleName = $this->formatName($request->getModuleName());
        $controllerName = $this->formatName($request->getControllerName(), true);
        $actionName = $this->formatName($request->getActionName());

        $request->setModuleName($moduleName);
        $request->setControllerName($controllerName);
        $request->setActionName($actionName);
    }

    public function fixProductFeed23009(Enlight_Event_EventArgs $args)
    {
        /** @var \Shopware\Models\ProductFeed\ProductFeed $entity */
        $entity = $args->get('entity');

        $version = $this->swVersion();

        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.4', '>=')
            || !$this->Config()->get('fix_authenticated_rce_product_export')
        ) {
            return;
        }

        $blacklist = array(
            'php',
            'php3',
            'php4',
            'php5',
            'phtml',
            'cgi',
            'pl',
            'sh',
            'com',
            'bat',
            '',
            'py',
            'rb',
        );

        $fileName = basename($entity->getFileName());
        $fileName = ltrim($fileName, '.');
        $ext = strtolower(pathinfo($fileName, \PATHINFO_EXTENSION));

        if (\in_array($ext, $blacklist, true)) {
            $fileName = str_replace('.' . $ext, '.invalid', strtolower($fileName));
        }

        $entity->setFileName($fileName);
    }

    public function fixPerformOrder23016(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();

        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.7', '>=')
            || !$this->Config()->get('fix_perform_order_23016')
        ) {
            return;
        }

        /** @var Enlight_Controller_Action $controller */
        $controller = $args->get('subject');

        if ($controller->Request()->getParam('hash') === sha1(false)) {
            throw new RuntimeException('Invalid hash given');
        }
    }

    public function fixXssOnForms23319(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.7', '>=')
            || !$this->Config()->get('fix_xss_23319')
        ) {
            return;
        }

        /** @var \Shopware_Controllers_Frontend_Forms $subject */
        $subject = $args->getSubject();
        $request = $subject->Request();

        $ordernumber = $request->getParam('sOrdernumber');

        if (!$ordernumber) {
            return;
        }

        if (!$this->isValidOrderNumber($ordernumber)) {
            $request->setParam('sOrdernumber', ''); // null doesn't work

            return;
        }

        $request->setParam('sOrdernumber', htmlspecialchars($ordernumber, \ENT_QUOTES | \ENT_SUBSTITUTE, 'utf-8'));
    }

    public function fixSessionFixation_23428(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.7', '>=')
            || !$this->Config()->get('fix_session_fixation_23428')
        ) {
            return;
        }

        if (\array_key_exists(session_name(), $_GET) || \array_key_exists(session_name(), $_POST)) {
            \Zend_Session::regenerateId();

            /** @var \Enlight_Controller_Front $front */
            $front = $args->getSubject();
            $front->Response()->setRedirect($front->Router()->assemble(array('controller' => 'index')));
        }
    }

    public function fixThemePreview_23428(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.7', '>=')
            || !$this->Config()->get('fix_session_fixation_23428')
        ) {
            return;
        }

        require_once __DIR__ . '/OptinService.php';
        $optinService = new \SwagSecurity\OptinService($this->get('dbal_connection'));

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        $themeId = (int) $subject->Request()->getParam('themeId');

        $shopId = (int) $subject->Request()->getParam('shopId');

        $session = $this->get('session');

        /** @var \Shopware\Models\Shop\Template $theme */
        $theme = $this->get('models')->getRepository('\Shopware\Models\Shop\Template')->find($themeId);

        /** @var \Shopware\Models\Shop\Shop $shop */
        $shop = $this->get('models')->getRepository('\Shopware\Models\Shop\Shop')->getActiveById($shopId);
        $shop->registerResources();

        $session->offsetSet('template', $theme->getTemplate());
        $session->offsetSet('Admin', true);

        if (!$subject->Request()->isXmlHttpRequest()) {
            $this->get('events')->notify('Shopware_Theme_Preview_Starts', array(
                'session' => $session,
                'shop' => $shop,
                'theme' => $theme,
            ));

            $themeToken = $optinService->add('swThemeToken', 120, array(
                'sessionName' => session_name(),
                'sessionValue' => $session->get('sessionId'),
            ));

            $url = $subject->Front()->Router()->assemble(array(
                'module' => 'frontend',
                'controller' => 'index',
                'themeToken' => $themeToken,
            ));

            $subject->redirect($url);
        }

        return true;
    }

    public function fixThemePreviewRestore_23428(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.7', '>=')
            || !$this->Config()->get('fix_session_fixation_23428')
        ) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        $themeToken = $subject->Request()->getParam('themeToken');

        if (!$themeToken) {
            return;
        }

        require_once __DIR__ . '/OptinService.php';
        $optinService = new \SwagSecurity\OptinService($this->get('dbal_connection'));

        $data = $optinService->get('swThemeToken', $themeToken);

        if ($data === null) {
            return;
        }

        $optinService->delete('swThemeToken', $themeToken);

        $subject->Response()->setCookie(
            $data['sessionName'],
            $data['sessionValue'],
            0,
            $subject->Request()->getBaseUrl(),
            null,
            $subject->Request()->isSecure(),
            true
        );

        // Disable http cache for this Request
        $subject->Response()->setHeader('Cache-Control', 'private', true);

        $subject->redirect(array('controller' => 'index', 'action' => 'index'));

        return true;
    }

    public function fixAjaxCalls_23428(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.2.0', '>=')
            || !$this->Config()->get('fix_session_fixation_23428')
        ) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        $subject->View()->addTemplateDir($this->Path() . '/Views/');
        $subject->View()->extendsTemplate('frontend/statistics_fix.tpl');
    }

    public function fixXSS_24473(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if (
            $version !== '5.6.0'
            || !$this->Config()->get('fix_xss_24473')
        ) {
            return;
        }

        /** @var Enlight_Controller_Request_RequestHttp $request */
        $request = $args->get('request');
        $request->query->replace($_GET);
        $request->request->replace($_REQUEST);
        $request->cookies->replace($_COOKIE);
        $request->server->replace($_SERVER);
    }

    public function onCollectJavaScriptFiles23007()
    {
        $version = $this->swVersion();
        if (
            version_compare($version, '5.0.0', '<')
            || version_compare($version, '5.5.7', '>=')
            || !$this->Config()->get('fix_xsrf_token_leakage_23007')
        ) {
            return;
        }

        return new ArrayCollection(array(
            __DIR__ . '/Views/frontend/js/jquery.csrf-fix.js',
        ));
    }

    public function includeNotifierJavascript(Enlight_Event_EventArgs $args)
    {
        /** @var \Shopware_Controllers_Backend_Index $subject */
        $subject = $args->getSubject();

        $subject->View()->addTemplateDir(__DIR__ . '/Views');

        if ($subject->Request()->getActionName() === 'load') {
            $subject->View()->extendsTemplate('backend/swag_security/window.js');
        }
    }

    public function fixInvalidDOIUser_24590(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();

        if (
            version_compare($version, '5.4.5', '<')
            || version_compare($version, '5.6.2', '>=')
            || !$this->Config()->get('fix_invalid_doi_user_24590')
        ) {
            return;
        }

        $context = $args->getReturn();
        $confirmLink = $context['sConfirmLink'];

        $explodedConfirmLink = explode('/', $confirmLink);
        $hash = array_reverse($explodedConfirmLink)[0];
        $mail = $args->get('mail');
        $customerIdByMail = $this->getCustomerIdByMail($mail);

        if ($customerIdByMail === $this->getCustomerIdByOptInHash($hash)) {
            return;
        }

        $confirmLink = str_replace($hash, $this->getHashForCustomerId($customerIdByMail), $confirmLink);
        $context['sConfirmLink'] = $confirmLink;

        return $context;
    }

    public function fixCachedBlogComment_25409(Enlight_Event_EventArgs $args)
    {
        if ($this->fixCachedBlogComment_25409_inactive()) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        $subject->View()->addTemplateDir($this->Path() . '/Views/');
    }

    public function registerController_25409(Enlight_Event_EventArgs $args)
    {
        if ($this->fixCachedBlogComment_25409_inactive()) {
            return null;
        }

        $this->get('template')->addTemplateDir(sprintf('%s/Views/', $this->Path()));

        return sprintf('%s/Controllers/Widgets/Blog.php', $this->Path());
    }

    public function escapeCustomerMailInCustomerModule_25773(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix25773()) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        if ($subject->Request()->getActionName() !== 'load') {
            return;
        }

        $subject->View()->addTemplateDir(sprintf('%s/Views/', $this->Path()));
        $subject->View()->extendsTemplate('backend/sw25773/customer/list.js');
        $subject->View()->extendsTemplate('backend/sw25773/customer/preview.js');
    }

    public function preventJavascriptExec_25772(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix25772()) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        if ($subject->Request()->getActionName() !== 'load') {
            return;
        }

        $subject->View()->addTemplateDir(sprintf('%s/Views/', $this->Path()));
        $subject->View()->extendsTemplate('backend/sw25772/newsletter_manager/editor.js');
    }

    public function filterNewsletterTitle_25772(Enlight_Hook_HookArgs $args)
    {
        if ($this->skipFix25772()) {
            return;
        }

        $return = $args->getReturn();

        if (!isset($return['containers'])) {
            return;
        }

        foreach ($return['containers'] as &$container) {
            if ($container['type'] === 'ctText') {
                $container['description'] = htmlspecialchars($container['description']);
            }
        }
        unset($container);

        $args->setReturn($return);
    }

    public function preventXssInEmotionElements_25771(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix25771()) {
            return;
        }

        $args->getSubject()->View()->addTemplateDir(__DIR__ . '/Views/');
    }

    public function preventXssInYoutubeElement_25771(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix25771()) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        if ($subject->Request()->getActionName() !== 'load') {
            return;
        }

        $subject->View()->addTemplateDir(sprintf('%s/Views/', $this->Path()));
        $subject->View()->extendsTemplate('backend/sw25771/emotion/youtube.js');
    }

    public function preventJsonResponseInErrorController_26001(Enlight_Controller_ActionEventArgs $args)
    {
        if (
            version_compare($this->swVersion(), '5.6.10', '>=')
            || !$this->Config()->get('json_response_in_error_controller_26001')
        ) {
            return;
        }

        if (!($args->getSubject() instanceof Shopware_Controllers_Frontend_Error)) {
            return;
        }

        if ($args->getRequest()->getServer('HTTP_CONTENT_TYPE', $args->getRequest()->getServer('CONTENT_TYPE')) !== 'application/json') {
            return;
        }

        if ($server = $args->getSubject()->Request()->server) {
            $server->set('CONTENT_TYPE', '');
            $server->set('HTTP_CONTENT_TYPE', '');
        } else {
            $args->getSubject()->Request()->setHeader('Content-Type', '');
        }
    }

    public function adjustGetTextImplementation_26050(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix26050()) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        if (
            $subject->Request()->getActionName() !== 'index'
            || $subject->Request()->getParam('file') !== 'app'
        ) {
            return;
        }

        $subject->View()->addTemplateDir(sprintf('%s/Views/', $this->Path()));
        $subject->View()->extendsTemplate('backend/sw26050/index/app.js');
    }

    public function escapeProductNameInProductModule_26050(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix26050()) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');
        if ($subject->Request()->getActionName() !== 'load') {
            return;
        }

        $subject->View()->addTemplateDir(sprintf('%s/Views/', $this->Path()));
        $subject->View()->extendsTemplate('backend/sw26050/article/controller/detail.js');
        $subject->View()->extendsTemplate('backend/sw26050/article/view/detail/window.js');
    }

    public function escapeProductNameOnDetailPage_26050(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        if ($this->skipFix26050() || version_compare($version, '5.7.0', '>')) {
            return;
        }

        /** @var Enlight_Controller_Action $subject */
        $subject = $args->get('subject');

        $subject->View()->addTemplateDir($this->Path() . '/Views/');
    }

    /**
     * @return void
     */
    public function addPasswordChangeDateToSessionUponPasswordChange_26448(Enlight_Controller_ActionEventArgs $args)
    {
        if ($this->skipFix26448()) {
            return;
        }

        /** @var Enlight_Controller_Request_Request $request */
        $request = $args->get('request');

        if ($request->getActionName() === 'savePassword') {
            $this->setPasswordChangedDate();
        }
    }

    /**
     * @return void
     */
    public function addPasswordChangeDateToSessionUponSuccessfulLogin_26448(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix26448()) {
            return;
        }

        $customerData = $args->get('user');

        if (!\is_array($customerData)) {
            return;
        }

        $this->setPasswordChangedDate($customerData['id']);
    }

    public function fixAclResource_26913(Enlight_Controller_ActionEventArgs $args)
    {
        if ($this->skipFix26913()) {
            return;
        }

        $request = $args->getRequest();

        if ($request->getModuleName() !== 'backend') {
            return;
        }

        $camelCasedName = preg_replace_callback('/(_|\.)+(.)/', function ($match) {
            return ($match[1] === '.' ? '_' : '') . strtoupper($match[2]);
        }, $request->getControllerName());

        $request->setControllerName($camelCasedName);
    }

    public function fixSensitiveData_26909(Enlight_Controller_ActionEventArgs $args)
    {
        if ($args->getRequest()->getActionName() !== 'getDetail') {
            return;
        }

        if ($this->skipFix26909()) {
            return;
        }

        /** @var \Shopware_Controllers_Frontend_Error $subject */
        $subject = $args->getSubject();
        $assigned = $subject->View()->getAssign();
        unset($assigned['data']['sessionId']);
        unset($assigned['data']['hashPassword']);

        $subject->View()->assign($assigned);
    }

    public function clearCsrfToken_26662(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();

        if (
            version_compare($version, '5.7.9', '>=')
            || version_compare($version, '5.2.0', '<')
            || !$this->Config()->get('fix_csrf_token_26662')
        ) {
            return;
        }

        $name = $this->getCsrfName();

        $shopContextService = $this->get('shopware_storefront.context_service');
        $shop = $shopContextService->getShopContext()->getShop();
        $path = $shop->getPath() ?: '';

        if (version_compare($this->swVersion(), '5.6.3', '>=')) {
            $path = sprintf('%s/', $path);
        }

        if ($path === '') {
            $path = '/';
        }

        setcookie(
            $name,
            Random::getAlphanumericString(30),
            0,
            $path,
            '',
            $shop->getSecure(),
            false
        );
    }

    /**
     * @return array|null
     */
    public function checkPasswordChangeDateInCheckUser_26448(Enlight_Event_EventArgs $args)
    {
        if ($this->skipFix26448()) {
            return null;
        }

        if ($args->getReturn() === array()) {
            return null;
        }

        /** @var Enlight_Components_Session_Namespace $session */
        $session = $this->get('session');

        $userId = $session->offsetGet('sUserId');
        $userMail = $session->offsetGet('sUserMail');
        $passwordChangeDate = $session->offsetGet('sUserPasswordChangeDate');

        if ($passwordChangeDate === null) {
            return array();
        }

        /** @var Enlight_Components_Db_Adapter_Pdo_Mysql $db */
        $db = $this->get('db');
        $sql = 'SELECT 1 FROM s_user WHERE password_change_date = ? AND email = ? AND id = ? AND UNIX_TIMESTAMP(lastlogin) >= (UNIX_TIMESTAMP(NOW())-?);';

        $getUser = $db->fetchOne(
            $sql,
            array(
                $passwordChangeDate,
                $userMail,
                $userId,
                (int) \ini_get('session.gc_maxlifetime'),
            )
        );

        if ($getUser !== false) {
            return null;
        }

        return array();
    }

    public function invalidateExistingTokensUponTokenCreation_26666(Enlight_Controller_ActionEventArgs $args)
    {
        if ($this->skipFix26666()) {
            return;
        }

        /** @var Enlight_Controller_Request_Request $request */
        $request = $args->get('request');
        $email = $request->getParam('email');

        if ($request->getActionName() !== 'password' || $request->getMethod() !== Request::METHOD_POST || empty($email)) {
            return;
        }

        $customerId = Shopware()->Modules()->Admin()->sGetUserByMail($email);

        /** @var Connection $connection */
        $connection = $this->get('dbal_connection');
        $querybuilder = $connection->createQueryBuilder();

        $querybuilder->update('s_core_optin')
            ->set('datum', ':datum')
            ->where($querybuilder->expr()->in('type', ':types'))
            ->andWhere($querybuilder->expr()->eq('data', ':customerId'))
            ->setParameters(array(
                'datum' => (new DateTime('-1 day'))->format(\DATE_ATOM),
                'customerId' => $customerId,
            ))
            ->setParameter('types', array('swPassword', 'password'), Connection::PARAM_STR_ARRAY)
            ->execute();
    }

    public function fixXss_26748(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        $fix = (version_compare($version, '5.7.12', '<') && $this->Config()->get('fix_xss_26748'));

        if (!$fix) {
            return;
        }

        $classLoader = new ClassLoader();
        $classLoader->register(true);

        $classLoader->addPsr4(
            'voku\\helper\\',
            array(
                __DIR__ . '/Engine/anti-xss/src/voku/helper',
                __DIR__ . '/Engine/portable-ascii/src/voku/helper/',
                __DIR__ . '/Engine/portable-utf8/src/voku/helper/',
        )
        );

        $request = $args->get('request');

        if ($request->getModuleName() === 'backend' || $request->getModuleName() === 'api') {
            return;
        }

        $userParams = $request->getUserParams();
        $process = array(
            &$_GET, &$_POST, &$_COOKIE, &$_REQUEST, &$_SERVER, &$userParams,
        );

        $route = strtolower(
            implode(
                '/',
                array($request->getModuleName(), $request->getControllerName(), $request->getActionName())
            )
        );

        $allowanceList = \array_key_exists($route, $this->allowanceList) ? $this->allowanceList[$route] : array();
        foreach ($process as $key => $val) {
            foreach ($val as $k => $v) {
                unset($process[$key][$k]);
                $allowedHtmlTags = \array_key_exists($k, $allowanceList) ? $allowanceList[$k][self::ALLOWED_HTML_TAGS_KEY] : array();
                $allowedAttributes = \array_key_exists($k, $allowanceList) ? $allowanceList[$k][self::ALLOWED_ATTRIBUTES_KEY] : array();

                if (\is_string($k)) {
                    $filteredKey = $this->useAntiXss($k, array(), array());
                } else {
                    $filteredKey = $k;
                }

                if ($filteredKey === '' || $filteredKey === null) {
                    continue;
                }

                if (\is_array($v)) {
                    $process[$key][$filteredKey] = $this->filterArrayValue($v, function ($value) use ($allowedHtmlTags, $allowedAttributes) {
                        return $this->useAntiXss($value, $allowedHtmlTags, $allowedAttributes);
                    });
                    continue;
                }

                $process[$key][$k] = $this->useAntiXss($v, $allowedHtmlTags, $allowedAttributes);
            }
        }

        unset($process);
        $request->setParams($userParams);

        if (version_compare($version, '5.6.0', '>=')) {
            $request->query->replace($_GET);
            $request->request->replace($_POST);
            $request->cookies->replace($_COOKIE);
            $request->server->replace($_SERVER);
        }
    }

    public function fixArraysXss_26657(Enlight_Event_EventArgs $args)
    {
        $version = $this->swVersion();
        $fix = (version_compare($version, '5.7.9', '<') && $this->Config()->get('fix_array_xss_26657'));

        if (!$fix) {
            return;
        }

        $request = $args->get('request');

        if ($request->getModuleName() === 'backend' || $request->getModuleName() === 'api') {
            return;
        }

        $regex = Shopware()->Plugins()->Frontend()->InputFilter()->xssRegex;

        $userParams = $request->getUserParams();
        $process = array(
            &$_GET, &$_POST, &$_COOKIE, &$_REQUEST, &$_SERVER, &$userParams,
        );

        $stripTagsConf = $this->collection->getConfig('InputFilter')->get('strip_tags', false);

        $route = strtolower(
            implode(
                '/',
                array($request->getModuleName(), $request->getControllerName(), $request->getActionName())
            )
        );

        $stripTagsWhiteList = \array_key_exists($route, $this->getKeyWhitelist()) ? $this->getKeyWhitelist()[$route] : array();
        foreach ($process as $key => $val) {
            foreach ($val as $k => $v) {
                unset($process[$key][$k]);
                $stripTags = \in_array($k, $stripTagsWhiteList) ? false : $stripTagsConf;

                if (\is_string($k)) {
                    $filteredKey = Shopware_Plugins_Frontend_InputFilter_Bootstrap::filterValue($k, $regex, $stripTags);
                } else {
                    $filteredKey = $k;
                }

                if ($filteredKey === '' || $filteredKey === null) {
                    continue;
                }

                if (\is_array($v)) {
                    $process[$key][$filteredKey] = $this->filterArrayValue($v, function ($value) use ($regex, $stripTags) {
                        return Shopware_Plugins_Frontend_InputFilter_Bootstrap::filterValue($value, $regex, $stripTags);
                    });
                    continue;
                }

                $process[$key][$k] = $v;
            }
        }

        unset($process);
        $request->setParams($userParams);

        if (version_compare($version, '5.6.0', '>=')) {
            $request->query->replace($_GET);
            $request->request->replace($_POST);
            $request->cookies->replace($_COOKIE);
            $request->server->replace($_SERVER);
        }
    }

    /**
     * Strips invalid characters from controller, module and action
     *
     * @param string $unFormatted
     * @param bool   $isController
     *
     * @return string
     */
    protected function formatName($unFormatted, $isController = false)
    {
        $allowedCharacters = 'a-zA-Z0-9_';

        if ($isController) {
            $allowedCharacters .= '\.';
        }

        return preg_replace('#[^' . $allowedCharacters . ']+#', '', $unFormatted);
    }

    /**
     * @param string $orderNumber
     *
     * @return bool
     */
    protected function isValidOrderNumber($orderNumber)
    {
        // This regex needs to match with the one in the assert in \Shopware\Models\Article\Detail::$number
        return preg_match('/^[a-zA-Z0-9-_.]+$/', $orderNumber) === 1;
    }

    private function skipFix26909()
    {
        $version = $this->swVersion();

        return version_compare($version, '5.7.15', '>')
            || !$this->Config()->get('fixSensitiveData_26909');
    }

    /**
     * @return bool
     */
    private function skipFix26913()
    {
        $version = $this->swVersion();

        return version_compare($version, '5.7.15', '>')
            || !$this->Config()->get('fixAclResource_26913');
    }

    private function useAntiXss($value, $allowedHtmlTags, $allowedAttributes)
    {
        $antiXss = new AntiXSS();
        $antiXss->removeEvilAttributes($allowedAttributes);
        $antiXss->removeEvilHtmlTags($allowedHtmlTags);
        $value = $antiXss->xss_clean($value);

        return \str_replace(array('&lt;', '&gt;'), array('<', '>'), $value);
    }

    /**
     * @param array<string|int, mixed> $value
     *
     * @return array<string|int, mixed>|null
     */
    private function filterArrayValue($value, $callback)
    {
        $newReturn = array();
        foreach ($value as $valueKey => $valueValue) {
            if (\is_int($valueKey)) {
                $filteredKey = $valueKey;
            } else {
                $filteredKey = $callback($valueKey);
            }

            if ($filteredKey === '' || $filteredKey === null) {
                continue;
            }

            $filteredValue = $valueValue;

            if (\is_array($valueValue)) {
                $filteredValue = $this->filterArrayValue($valueValue, $callback);
            }

            if (\is_string($valueValue)) {
                $filteredValue = $callback($valueValue);
            }

            $newReturn[$filteredKey] = $filteredValue;
        }

        return $newReturn;
    }

    private function getKeyWhitelist()
    {
        return array(
            'frontend/account/login' => array(
                'password',
            ),
            'frontend/account/savepassword' => array(
                'password',
                'passwordConfirmation',
                'currentPassword',
            ),
            'frontend/register/ajax_validate_email' => array(
                'password',
            ),
            'frontend/register/ajax_validate_password' => array(
                'password',
            ),
            'frontend/register/saveregister' => array(
                'password',
            ),
            'frontend/account/resetpassword' => array(
                'password',
                'passwordConfirmation',
            ),
            'frontend/account/saveemail' => array(
                'currentPassword',
            ),
        );
    }

    /**
     * @param string|null $customerId
     *
     * @return void
     */
    private function setPasswordChangedDate($customerId = null)
    {
        /** @var Enlight_Components_Session_Namespace $session */
        $session = $this->get('session');
        $customerId = $customerId ?: $session->get('sUserId');

        if ($customerId === null) {
            return;
        }

        /** @var Connection $db */
        $db = $this->get('dbal_connection');
        $passwordChangeDate = $db->fetchOne(
            'SELECT `password_change_date` FROM `s_user` WHERE `id` = :customerId;',
            array('customerId' => $customerId)
        );

        $session->offsetSet('sUserPasswordChangeDate', $passwordChangeDate);
    }

    private function skipFix25773()
    {
        $version = $this->swVersion();
        $fixedAlready = version_compare($version, '5.3.0', '<')
            || (version_compare($version, '5.6.9', '>=') && version_compare($version, '5.6.10', '<='))
            || version_compare($version, '5.7.14', '>=');

        return $fixedAlready || !$this->Config()->get('xss_backend_customer_mail_25773');
    }

    private function skipFix25772()
    {
        $version = $this->swVersion();
        $fixedAlready = (version_compare($version, '5.6.9', '>=') && version_compare($version, '5.6.10', '<='))
            || version_compare($version, '5.7.14', '>=');

        return $fixedAlready || !$this->Config()->get('xss_backend_newsletter_25772');
    }

    private function skipFix25771()
    {
        $version = $this->swVersion();
        $fixedAlready = (version_compare($version, '5.6.9', '>=') && version_compare($version, '5.6.10', '<='))
            || version_compare($version, '5.7.14', '>=');

        return $fixedAlready || !$this->Config()->get('xss_backend_emotion_elements_25771');
    }

    /**
     * @return bool
     */
    private function skipFix26050()
    {
        $version = $this->swVersion();
        $fixedAlready = version_compare($version, '5.6.10', '==')
            || version_compare($version, '5.7.14', '>=');

        return $fixedAlready || !$this->Config()->get('xss_backend_product_name_26050');
    }

    /**
     * @return bool
     */
    private function skipFix26448()
    {
        $version = $this->swVersion();

        return version_compare($version, '5.7.3', '<')
            || version_compare($version, '5.7.7', '>=')
            || !$this->Config()->get('session_invalidation_upon_password_change_26448');
    }

    /**
     * @return bool
     */
    private function skipFix26666()
    {
        $version = $this->swVersion();

        return version_compare($version, '5.2.21', '<')
            || version_compare($version, '5.7.9', '>=')
            || !$this->Config()->get('password_reset_token_invalidation_26666');
    }

    private function fixCachedBlogComment_25409_inactive()
    {
        $version = $this->swVersion();

        return version_compare($version, '5.6.7', '>=') || !$this->Config()->get('fix_cached_blog_comment_25409');
    }

    /**
     * @param int $customerId
     *
     * @return string
     */
    private function getHashForCustomerId($customerId)
    {
        /** @var Connection $connection */
        $connection = $this->get('dbal_connection');
        $sql = <<<'SQL'
            SELECT hash
            FROM s_core_optin optin
            WHERE optin.data LIKE :customerIdString
SQL;

        $customerIdSearchString = '%"customerId";i:' . $customerId . ';%';

        $stmt = $connection->prepare($sql);
        $stmt->bindParam(':customerIdString', $customerIdSearchString);
        $stmt->execute();

        return $stmt->fetchColumn();
    }

    /**
     * @param string $mail
     *
     * @return int int
     */
    private function getCustomerIdByMail($mail)
    {
        /** @var Connection $connection */
        $connection = $this->get('dbal_connection');
        $sql = <<<'SQL'
            SELECT id
            FROM s_user `user`
            WHERE `user`.email = :mail
SQL;

        $stmt = $connection->prepare($sql);
        $stmt->bindParam(':mail', $mail);
        $stmt->execute();

        return $stmt->fetchColumn();
    }

    /**
     * @param string $hash
     *
     * @return int int
     */
    private function getCustomerIdByOptInHash($hash)
    {
        /** @var Connection $connection */
        $connection = $this->get('dbal_connection');
        $sql = <<<'SQL'
            SELECT data
            FROM s_core_optin optin
            WHERE optin.hash = :hash
SQL;

        $stmt = $connection->prepare($sql);
        $stmt->bindParam(':hash', $hash);
        $stmt->execute();

        $optinData = unserialize($stmt->fetchColumn());

        return (int) $optinData['customerId'];
    }

    /**
     * Deletes an event from s_core_subscribes
     *
     * @param string $eventName
     */
    private function deleteEvent($eventName)
    {
        try {
            /** @var \Doctrine\DBAL\Connection $connection */
            $connection = $this->get('dbal_connection');
            $connection->delete('s_core_subscribes', array('listener' => $eventName));
        } catch (\Exception $e) {
        }
    }

    private function swVersion($type = 'full')
    {
        return ShopwareVersion::get($type);
    }

    /**
     * @return array
     */
    private function castIntArray($ids)
    {
        if (!\is_array($ids)) {
            $ids = array($ids);
        }
        $ids = array_map(function ($id) {
            return (int) $id;
        }, $ids);

        return $ids;
    }

    /**
     * @return array
     */
    private function getSmartySecurityConfig()
    {
        $user = array();
        try {
            $user = Shopware()->getOption('template_security');
        } catch (\Exception $e) {
        }

        if (empty($user)) {
            try {
                $user = Shopware()->Container()->getParameter('shopware.template_security');
            } catch (\Exception $e) {
            }
        }

        $default = array(
            'php_modifiers' => include __DIR__ . '/smarty_functions.php',
            'php_functions' => include __DIR__ . '/smarty_functions.php',
        );

        if (empty($user)) {
            return $default;
        }

        return array_merge_recursive($default, $user);
    }

    /**
     * @param Enlight_Controller_Request_Request $request
     */
    private function cleanValues($request)
    {
        $dangerousFields = array('company', 'department', 'salutation', 'firstname', 'zipcode', 'city', 'phone', 'vatid',
            'additionaladdressline1', 'additionaladdressline2', 'title', 'firstname', 'lastname', 'street', 'zipcode',
            'email', 'referer', 'internalcomment', 'lastname', 'number', 'transactionid', 'comment', 'customercomment',
            'temporaryid', 'languageiso', 'currency', 'remoteaddress', 'partner', 'devicetype', 'contact', 'fax',
            'countryname', 'web', 'profile', );

        $params = $request->getPost();
        $this->filterParams($dangerousFields, $params, $request);

        $request->setPost($params);
    }

    /**
     * @param array                              $dangerousFields
     * @param array                              $params
     * @param Enlight_Controller_Request_Request $request
     */
    private function filterParams($dangerousFields, &$params, $request)
    {
        foreach ($params as $fieldName => &$value) {
            if (\is_array($value)) {
                $this->filterParams($dangerousFields, $value, $request);
                continue;
            }

            if (\is_string($value) && \in_array(strtolower($fieldName), $dangerousFields)) {
                $params[$fieldName] = strip_tags($params[$fieldName]);
            }
        }
    }

    /**
     * Helper method useful for validating the column names to prevent SQL injections
     *
     * @param string $field
     *
     * @throws \Exception
     */
    private function validateField($field)
    {
        if (\strlen($field) > 64) {
            throw new \Exception('Maximum length: 64 chars');
        }

        if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $field)) {
            throw new \Exception(sprintf('Invalid chars in %s', $field));
        }
    }

    /**
     * Helper method useful for validating the content of default values to prevent SQL injections
     *
     * @param float|int|string|null $defaultValue
     *
     * @return float|int|string|null
     */
    private function filterDefaultValue($defaultValue)
    {
        if (!\is_string($defaultValue)) {
            return $defaultValue;
        }

        if ($defaultValue === 'NULL') {
            return $defaultValue;
        }

        $connection = $this->get('dbal_connection');

        return $connection->quote(
            str_replace(array('`', '\''), '', $defaultValue)
        );
    }

    /**
     * @param int $currencyId
     */
    private function updateCurrencyDependencies($currencyId)
    {
        /** @var \Shopware\Models\Shop\Currency $currencyModel */
        $currencyModel = $this->get('models')->find('Shopware\Models\Shop\Currency', $currencyId);

        /** @var Shopware\Models\Shop\Shop $shopModel */
        $shopModel = $this->get('shop');
        $shopModel->setCurrency($currencyModel);

        /** @var \Zend_Currency $currency */
        $currency = $this->get('Currency');
        $currency->setFormat($currencyModel->toArray());

        if ($this->get('service_container')->has('shopware_storefront.context_service')) {
            $this->get('shopware_storefront.context_service')->initializeShopContext();
        }
    }

    /**
     * @param int $userId
     *
     * @return string
     */
    private function getUserMailById($userId)
    {
        /** @var \Doctrine\DBAL\Query\QueryBuilder $queryBuilder */
        $queryBuilder = $this->get('dbal_connection')->createQueryBuilder();

        return (string) $queryBuilder->select('user.email')
            ->from('s_user', 'user')
            ->where('user.id = :userId')
            ->setParameter(':userId', $userId)
            ->execute()
            ->fetchColumn();
    }

    /**
     * @throws Exception
     *
     * @return array
     */
    private function getCurrencyData()
    {
        /** @var \Shopware\Models\Shop\Currency $currencyModel */
        $currencyModel = $this->get('shop')->getCurrency();

        if (!$currencyModel) {
            throw new \Exception('No shop context/currency available, which is required for the checkout process. This request has therefore been denied. This is enforced by the fix 21593 from the SwagSecurityPlugin');
        }

        $currency = array();

        // Shopware 4.3 compatibility
        $currency['sCurrencyId'] = $currencyModel->getId();
        $currency['sCurrencyName'] = $currencyModel->getCurrency();
        $currency['sCurrencyFactor'] = $currencyModel->getFactor();

        return $currency;
    }

    /**
     * Adds .htaccess-Files to directories which have been unprotected in the
     * past.
     */
    private function addCacheDirectoryHtaccess()
    {
        $container = Shopware()->Container();
        $cacheDir = \dirname($container->getParameter('kernel.cache_dir'));

        $cachePath = $cacheDir . \DIRECTORY_SEPARATOR . '.htaccess';
        $backupPath = Shopware()->DocPath('files_backup') . \DIRECTORY_SEPARATOR . '.htaccess';

        $files = array(
            $cachePath,
            $backupPath,
        );

        foreach ($files as $file) {
            if (!is_dir(\dirname($file))) {
                continue;
            }

            if (!file_exists($file)) {
                $success = copy($this->Path() . '/Assets/sw-22811.htaccess', $file);

                if (!$success) {
                    throw new RuntimeException(
                        sprintf('Cannot copy %s to %s', $this->Path() . '/Assets/sw-22811.htaccess', $file)
                    );
                }
            }
        }
    }

    /**
     * @return string
     */
    private function getCsrfName()
    {
        $shopContextService = $this->get('shopware_storefront.context_service');
        $componentsConfig = $this->get('config');

        $context = $shopContextService->getShopContext();

        $name = '__csrf_token-' . $context->getShop()->getId();

        if ($context->getShop()->getParentId() && $componentsConfig->get('shareSessionBetweenLanguageShops')) {
            $name = '__csrf_token-' . $context->getShop()->getParentId();
        }

        return $name;
    }
}
