<?php
/**
 * Shopware 5
 * Copyright (c) shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 */

namespace SwagSecurity\Components\Model;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Query\Expr\Comparison;
use Shopware\Components\Model\QueryBuilder as BaseQueryBuilder;
use Shopware\Components\Model\QueryOperatorValidator;

/**
 * The Shopware QueryBuilder is an extension of the standard Doctrine QueryBuilder.
 */
class QueryBuilder extends BaseQueryBuilder
{
    /**
     * @var QueryOperatorValidator
     */
    protected $operatorValidator;

    public function __construct(EntityManagerInterface $em, QueryOperatorValidator $operatorValidator)
    {
        $this->operatorValidator = $operatorValidator;

        parent::__construct($em);
    }

    /**
     * Adds filters to the query results.
     *
     * <code>
     *      $this->addFilter(array(
     *          'name' => 'A%'
     *      ));
     * </code>
     *
     * <code>
     *      $this->addFilter(array(array(
     *          'property' => 'name'
     *          'value' => 'A%'
     *      )));
     * </code>
     *
     * <code>
     *      $this->addFilter(array(array(
     *          'property'   => 'number'
     *          'expression' => '>',
     *          'value'      => '500'
     *      )));
     * </code>
     *
     * @return QueryBuilder
     */
    public function addFilter(array $filter)
    {
        foreach ($filter as $exprKey => $where) {
            if (\is_object($where)) {
                $this->andWhere($where);
                continue;
            }

            $operator = null;
            $expression = null;

            if (\is_array($where) && isset($where['property'])) {
                $exprKey = $where['property'];

                if (isset($where['expression']) && !empty($where['expression'])) {
                    $expression = $where['expression'];
                }

                if (isset($where['operator']) && !empty($where['operator'])) {
                    $operator = $where['operator'];
                }

                $where = $where['value'];
            }

            if (!\preg_match('#^[a-z][a-z0-9_.]+$#i', $exprKey)) {
                continue;
            }

            // The return value of uniqid, even w/o parameters, may contain dots in some environments
            // so we make sure to strip those as well
            $parameterKey = \str_replace(array('.'), array('_'), $exprKey . \uniqid());
            if (isset($this->alias) && \strpos($exprKey, '.') === false) {
                $exprKey = $this->alias . '.' . $exprKey;
            }

            if ($expression == null) {
                switch (true) {
                    case \is_string($where):
                        $expression = 'LIKE';
                        break;

                    case \is_array($where):
                        $expression = 'IN';
                        break;

                    case $where === null:
                        $expression = 'IS NULL';
                        break;

                    default:
                        $expression = '=';
                        break;
                }
            }

            if ($where === null) {
                $expression = 'IS NULL';
            }

            $exprParameterKey = ':' . $parameterKey;
            if (\is_array($where)) {
                $exprParameterKey = '(' . $exprParameterKey . ')';
            }

            $this->operatorValidator->isValid($expression);
            $expression = new Comparison($exprKey, $expression, $where !== null ? $exprParameterKey : null);

            if (isset($operator)) {
                $this->orWhere($expression);
            } else {
                $this->andWhere($expression);
            }

            if ($where !== null) {
                $this->setParameter($parameterKey, $where);
            }
        }

        return $this;
    }
}
