<?php
/**
 * Shopware 5
 * Copyright (c) shopware AG
 *
 * According to our dual licensing model, this program can be used either
 * under the terms of the GNU Affero General Public License, version 3,
 * or under a proprietary license.
 *
 * The texts of the GNU Affero General Public License with an additional
 * permission and of our proprietary license can be found at and
 * in the LICENSE file you have received along with this program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * "Shopware" is a registered trademark of shopware AG.
 * The licensing of the program under the AGPLv3 does not imply a
 * trademark license. Therefore any rights, title and interest in
 * our trademarks remain entirely with us.
 */

namespace SwagSecurity;

use Doctrine\DBAL\Connection;
use Shopware\Components\Random;

class OptinService
{
    /**
     * @var Connection
     */
    private $connection;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * @param string $type
     * @param int    $duration
     *
     * @return string
     */
    public function add($type, $duration, array $data)
    {
        $hash = Random::getAlphanumericString(32);

        $this->connection->insert('s_core_optin', array(
            'type' => $type,
            'datum' => date('Y-m-d H:i:s', time() + $duration),
            'hash' => $hash,
            'data' => serialize($data),
        ));

        return $hash;
    }

    /**
     * @param string $type
     * @param string $hash
     *
     * @return array|null
     */
    public function get($type, $hash)
    {
        $data = $this->connection->createQueryBuilder()->from('s_core_optin', 'optin')
            ->select('data')
            ->where('optin.hash = :hash')
            ->andWhere('optin.type = :type')
            ->andWhere('optin.datum >= :currentDate')
            ->setParameter('hash', $hash)
            ->setParameter('type', $type)
            ->setParameter('currentDate', date('Y-m-d H:i:s'))
            ->execute()
            ->fetchColumn();

        if (empty($data)) {
            return null;
        }

        if (\PHP_MAJOR_VERSION < 7) {
            return unserialize($data);
        }

        // phpcs:disable
        return unserialize($data, array('allowed_classes' => false));
        // phpcs:enable
    }

    /**
     * @param string $type
     * @param string $hash
     */
    public function delete($type, $hash)
    {
        $this->connection->createQueryBuilder()
            ->delete('s_core_optin')
            ->where('hash = :hash')
            ->andWhere('type = :type')
            ->setParameter('hash', $hash)
            ->setParameter('type', $type)
            ->execute();
    }
}
