<?php
/**
 * (c) shopware AG <info@shopware.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

use Shopware\Bundle\MediaBundle\MediaService;
use Shopware\Components\CSRFWhitelistAware;
use Shopware\Components\Logger;
use Shopware\Components\Model\ModelManager;
use Shopware\Models\Customer\Address;
use Shopware\Models\Customer\Customer;

require_once __DIR__ . '/../../Components/CSRFWhitelistAware.php';

class Shopware_Controllers_Frontend_PaymentPaypal extends Shopware_Controllers_Frontend_Payment implements CSRFWhitelistAware
{
    /**
     * @var Shopware_Plugins_Frontend_SwagPaymentPaypal_Bootstrap
     */
    private $plugin;

    /**
     * @var Enlight_Components_Session_Namespace
     */
    private $session;

    /**
     * Whitelist notify- and webhook-action for paypal
     */
    public function getWhitelistedCSRFActions()
    {
        return array(
            'notify',
            'webhook',
        );
    }

    /**
     * {@inheritdoc}
     */
    public function init()
    {
        $this->plugin = $this->get('plugins')->Frontend()->SwagPaymentPaypal();
        $this->session = $this->get('session');
    }

    /**
     * {@inheritdoc}
     */
    public function get($name)
    {
        if (defined('Shopware::VERSION') && version_compare(Shopware::VERSION, '4.2.0', '<') && Shopware::VERSION !== '___VERSION___') {
            if ($name === 'pluginlogger') {
                $name = 'log';
            }
            $name = ucfirst($name);

            return Shopware()->Bootstrap()->getResource($name);
        }

        return Shopware()->Container()->get($name);
    }

    /**
     * {@inheritdoc}
     */
    public function preDispatch()
    {
        if (in_array($this->Request()->getActionName(), array('recurring', 'notify'))) {
            $this->Front()->Plugins()->ViewRenderer()->setNoRender();
        }
    }

    /**
     * Returns if the current user is logged in
     *
     * @return bool
     */
    public function isUserLoggedIn()
    {
        $customerId = $this->session->offsetGet('sUserId');

        return $customerId !== null && !empty($customerId);
    }

    /**
     * Index action method.
     *
     * Forwards to correct the action.
     */
    public function indexAction()
    {
        // PayPal Express > Sale
        $payPalResponse = $this->session->offsetGet('PaypalResponse');
        if (!empty($payPalResponse['TOKEN'])) {
            $this->forward('return');
        // Paypal Basis || PayPal Express
        } elseif ($this->getPaymentShortName() === 'paypal') {
            $this->forward('gateway');
        } else {
            $this->redirect(array('controller' => 'checkout'));
        }
    }

    /**
     * Express payment action method.
     */
    public function expressAction()
    {
        unset($this->session->sOrderVariables);

        $payment = $this->plugin->getPayment();
        if ($payment !== null) {
            $this->session->offsetSet('sPaymentID', $payment->getId());
        }

        $this->forward('gateway', 'PaymentPaypal', 'frontend', array('express' => true));
    }

    /**
     * Gateway payment action method.
     *
     * Collects the payment information and transmit it to the payment provider.
     */
    public function gatewayAction()
    {
        $router = $this->Front()->Router();
        $config = $this->plugin->Config();
        $client = $this->get('paypalClient');
        $isExpressCheckout = $this->Request()->getParam('express', false);

        $logoImage = $config->get('paypalLogoImage');
        if ($logoImage !== null) {
            if ($this->plugin->isShopware51() && !$this->plugin->isShopware52()) {
                /** @var MediaService $mediaService */
                $mediaService = $this->get('shopware_media.media_service');
                $logoImage = $mediaService->getUrl($logoImage);
            }

            $logoImage = 'string:{link file=' . var_export($logoImage, true) . ' fullPath}';
            $logoImage = $this->View()->fetch($logoImage);
        }

        $shopName = $config->get('paypalBrandName') ?: Shopware()->Config()->get('shopName');

        $borderColor = ltrim($config->get('paypalCartBorderColor'), '#');
        $paymentAction = $config->get('paypalPaymentAction', 'Sale');
        $user = $this->getUser();

        $params = array(
            'PAYMENTREQUEST_0_PAYMENTACTION' => $paymentAction,
            'RETURNURL' => $router->assemble(array(
                'action' => 'return',
                'forceSecure' => true,
                'express' => $isExpressCheckout,
            )),
            'CANCELURL' => $router->assemble(array('action' => 'cancel', 'forceSecure' => true)),
            'PAYMENTREQUEST_0_NOTIFYURL' => $router->assemble(array(
                'action' => 'notify',
                'forceSecure' => true,
            )),
            'GIROPAYSUCCESSURL' => $router->assemble(array('action' => 'return', 'forceSecure' => true)),
            'GIROPAYCANCELURL' => $router->assemble(array('action' => 'cancel', 'forceSecure' => true)),
            'BANKTXNPENDINGURL' => $router->assemble(array('action' => 'return', 'forceSecure' => true)),
            'ALLOWNOTE' => 1,
            'ADDROVERRIDE' => $user === null ? 0 : 1,
            'BRANDNAME' => $shopName,
            'LOGOIMG' => $logoImage,
            'CARTBORDERCOLOR' => $borderColor,
            'PAYMENTREQUEST_0_CUSTOM' => $this->createPaymentUniqueId(),
            'TOTALTYPE' => $user !== null ? 'Total' : 'EstimatedTotal',
            );

        if ($config->get('paypalBillingAgreement')) {
            $params['BILLINGTYPE'] = 'MerchantInitiatedBilling';
        }

        $params = array_merge($params, $this->getBasketParameter());
        $params = array_merge($params, $this->getCustomerParameter());

        $params = $this->get('events')->filter(
            'Shopware_Controllers_Frontend_PaymentPaypal_Gateway_Params',
            $params,
            array('config' => $config, 'user' => $user)
        );

        $response = $client->setExpressCheckout($params);

        $this->session->offsetSet('PaypalResponse', $response);

        if ($response['ACK'] === 'SuccessWithWarning') {
            $response['ACK'] = 'Success';
        }

        if (!empty($response['ACK']) && $response['ACK'] === 'Success') {
            $gatewayUrl = 'https://www.paypal.com/cgi-bin/';
            if ((bool) $config->get('paypalSandbox')) {
                $gatewayUrl = 'https://www.sandbox.paypal.com/cgi-bin/';
            }
            $gatewayUrl .= 'webscr?cmd=_express-checkout';
            if ($user !== null) {
                $gatewayUrl .= '&useraction=commit';
            }
            $gatewayUrl .= '&token=' . urlencode($response['TOKEN']);
            $this->View()->assign('PaypalGatewayUrl', $gatewayUrl);
        } else {
            $this->forward('return');
        }
    }

    /**
     * Recurring payment action method.
     */
    public function recurringAction()
    {
        if (!$this->getAmount() || $this->getOrderNumber()) {
            $this->redirect(array('controller' => 'checkout'));

            return;
        }
        $orderId = $this->Request()->getParam('orderId');
        $sql = '
            SELECT swag_payal_billing_agreement_id
            FROM s_order_attributes a, s_order o
            WHERE o.id = a.orderID
            AND o.id = ?
            AND o.userID = ?
            AND a.swag_payal_billing_agreement_id IS NOT NULL
            ORDER BY o.id DESC
        ';
        $agreementId = $this->get('db')->fetchOne($sql, array($orderId, $this->session->offsetGet('sUserId')));
        $details = array('REFERENCEID' => $agreementId);
        $response = $this->finishCheckout($details);

        if ($this->Request()->isXmlHttpRequest()) {
            if ($response['ACK'] !== 'Success') {
                $data = array(
                    'success' => false,
                    'message' => "[{$response['PAYMENTINFO_0_ERRORCODE']}] - {$response['PAYMENTINFO_0_SHORTMESSAGE']}",
                );
            } else {
                $data = array(
                    'success' => false,
                    'data' => array(
                        array(
                            'orderNumber' => $response['PAYMENTREQUEST_0_INVNUM'],
                            'transactionId' => $response['PAYMENTREQUEST_0_TRANSACTIONID'],
                        ),
                    ),
                );
            }
            echo Zend_Json::encode($data);
        } else {
            if ($response['ACK'] !== 'Success') {
                $this->View()->loadTemplate('frontend/payment_paypal/return.tpl');
                $this->View()->assign('PaypalConfig', $this->plugin->Config());
                $this->View()->assign('PaypalResponse', $response);
            } else {
                $this->redirect(
                    array(
                        'controller' => 'checkout',
                        'action' => 'finish',
                        'sUniqueID' => $response['PAYMENTREQUEST_0_CUSTOM'],
                    )
                );
            }
        }
    }

    /**
     * Return action method
     *
     * Reads the transactionResult and represents it for the customer.
     */
    public function returnAction()
    {
        $token = $this->Request()->getParam('token');
        $config = $this->plugin->Config();
        $client = $this->get('paypalClient');
        $initialResponse = $this->session->offsetGet('PaypalResponse');
        $isExpressCheckout = $this->Request()->getParam('express', false);

        $details = array();
        if ($token !== null) {
            $details = $client->getExpressCheckoutDetails(array('token' => $token));
        } elseif (!empty($initialResponse['TOKEN'])) {
            $details = $client->getExpressCheckoutDetails(array('token' => $initialResponse['TOKEN']));
        }

        // Canceled payment
        if (isset($details['CHECKOUTSTATUS'])
            && !isset($details['PAYERID'], $details['PAYMENTREQUEST_0_ADDRESSSTATUS'])
        ) {
            unset($this->session->PaypalResponse);

            $this->forward('gateway');

            return;
        }

        if ($initialResponse['ACK'] === 'Failure') {
            $this->logError($initialResponse);
        }

        switch (!empty($details['CHECKOUTSTATUS']) ? $details['CHECKOUTSTATUS'] : null) {
            case 'PaymentActionCompleted':
            case 'PaymentCompleted':
                $this->redirect(
                    array(
                        'controller' => 'checkout',
                        'action' => 'finish',
                        'sUniqueID' => $details['PAYMENTREQUEST_0_CUSTOM'],
                    )
                );
                break;
            case 'PaymentActionNotInitiated':
                /*
                 * This procedure will be executed if the user has been redirected from the PayPal page back to the shop.
                 * It won't be executed if the customer confirms the payment on the shopware confirm page, because only the return link
                 * has an express parameter.
                 */
                if ($isExpressCheckout) {
                    if (!empty($details['PAYERID']) && !empty($details['PAYMENTREQUEST_0_SHIPTONAME']) && !$this->isUserLoggedIn()) {
                        $this->createAccount($details);
                    }

                    $this->redirect(array('controller' => 'checkout'));

                    break;
                }

                /**
                 * If the user exists and the order is not finished.
                 *
                 * Will ony be triggered during normal checkout as $this->session->sOrderVariables is
                 * filled during the checkout and not available during the express checkout
                 */
                if ($this->getUser() && $this->getOrderNumber() === null) {
                    unset($this->session->PaypalResponse);
                    $response = $this->finishCheckout($details);
                    if ($response['ACK'] !== 'Success') {
                        if ((int) $response['L_ERRORCODE0'] === 10486) {
                            $redirectUrl = 'https://www.paypal.com/cgi-bin/';
                            if ((bool) $config->get('paypalSandbox')) {
                                $redirectUrl = 'https://www.sandbox.paypal.com/cgi-bin/';
                            }
                            $redirectUrl .= 'webscr?cmd=_express-checkout';
                            $redirectUrl .= '&token=' . urlencode($details['TOKEN']);
                            $this->redirect($redirectUrl);
                        } else {
                            $this->View()->assign('PaypalConfig', $config);
                            $this->View()->assign('PaypalResponse', $response);

                            $this->logError($response);
                        }
                    } elseif ($response['REDIRECTREQUIRED'] === 'true') {
                        $redirectUrl = 'https://www.paypal.com/cgi-bin/';
                        if ((bool) $config->get('paypalSandbox')) {
                            $redirectUrl = 'https://www.sandbox.paypal.com/cgi-bin/';
                        }
                        $redirectUrl .= 'webscr?cmd=_complete-express-checkout';
                        $redirectUrl .= '&token=' . urlencode($response['TOKEN']);
                        $this->redirect($redirectUrl);
                    } else {
                        $this->redirect(
                            array(
                                'controller' => 'checkout',
                                'action' => 'finish',
                                'sUniqueID' => $response['PAYMENTREQUEST_0_CUSTOM'],
                            )
                        );
                    }
                }

                break;
            case 'PaymentActionInProgress':
            case 'PaymentActionFailed':
            default:
                $this->View()->assign('PaypalConfig', $config);
                $this->View()->assign('PaypalResponse', $initialResponse);
                $this->View()->assign('PaypalDetails', $details);
                unset($this->session->PaypalResponse);

                break;
        }
    }

    /**
     * Cancel action method
     */
    public function cancelAction()
    {
        unset($this->session->PaypalResponse);
        $config = $this->plugin->Config();
        $this->View()->assign('PaypalConfig', $config);
    }

    /**
     * Notify action method
     */
    public function notifyAction()
    {
        $txnId = $this->Request()->get('parent_txn_id') ?: $this->Request()->get('txn_id');
        try {
            $details = $this->get('paypalClient')->getTransactionDetails(array('TRANSACTIONID' => $txnId));
        } catch (Exception $e) {
            $message = sprintf(
                'PayPal-Notify: Exception %s',
                $e->getMessage()
            );
            $context = array('exception' => $e);
            $this->get('pluginlogger')->error($message, $context);
        }

        if (empty($details['PAYMENTSTATUS']) || empty($details['ACK']) || $details['ACK'] !== 'Success') {
            $message = sprintf(
                'PayPal-Notify: Could not find TRANSACTIONID %s',
                $txnId
            );
            $context = array('details' => $details);
            $this->get('pluginlogger')->error($message, $context);

            return;
        }

        $this->plugin->setPaymentStatus($details['TRANSACTIONID'], $details['PAYMENTSTATUS']);
    }

    /**
     * @return array
     */
    protected function finishCheckout(array $details)
    {
        $client = $client = $this->get('paypalClient');
        $config = $this->plugin->Config();

        $notifyUrl = $this->Front()->Router()->assemble(array('action' => 'notify', 'forceSecure' => true));

        $params = array(
            'TOKEN' => $details['TOKEN'],
            'PAYERID' => $details['PAYERID'],
            'PAYMENTREQUEST_0_NOTIFYURL' => $notifyUrl,
            'PAYMENTREQUEST_0_CUSTOM' => $details['PAYMENTREQUEST_0_CUSTOM'],
            'BUTTONSOURCE' => 'Shopware_Cart_ECS',
        );
        if (!empty($details['REFERENCEID'])) {
            $params = array(
                'REFERENCEID' => $details['REFERENCEID'],
                'IPADDRESS' => $this->Request()->getClientIp(),
                'PAYMENTREQUEST_0_NOTIFYURL' => $notifyUrl,
                'PAYMENTREQUEST_0_CUSTOM' => $this->createPaymentUniqueId(),
                'BUTTONSOURCE' => 'Shopware_Cart_ECM',
            );
        }

        if (!defined('Shopware::VERSION') || (Shopware::VERSION === '___VERSION___' || version_compare(Shopware::VERSION, '4.4.0') >= 0)) {
            $params['BUTTONSOURCE'] = 'Shopware_Cart_5';
        }

        if (empty($params['TOKEN']) && empty($params['REFERENCEID'])) {
            $params['PAYMENTREQUEST_0_PAYMENTACTION'] = 'Authorization';
        } else {
            $params['PAYMENTREQUEST_0_PAYMENTACTION'] = $config->get('paypalPaymentAction', 'Sale');
        }

        $params = array_merge($params, $this->getBasketParameter());
        $params = array_merge($params, $this->getCustomerParameter());

        if ($config->get('paypalSendInvoiceId')) {
            $orderNumber = $this->saveOrder(
                isset($params['TOKEN']) ? $params['TOKEN'] : $params['REFERENCEID'],
                $params['PAYMENTREQUEST_0_CUSTOM']
            );
            $prefix = $config->get('paypalPrefixInvoiceId');
            $params['PAYMENTREQUEST_0_INVNUM'] = $orderNumber;
            if (!empty($prefix)) {
                // Set prefixed invoice id - Remove special chars and spaces
                $prefix = str_replace(' ', '', $prefix);
                $prefix = preg_replace('/[^A-Za-z0-9\_]/', '', $prefix);
                $params['PAYMENTREQUEST_0_INVNUM'] = $prefix . $orderNumber;
            }
        }

        //$params['SOFTDESCRIPTOR'] = $orderNumber;

        if (!empty($params['REFERENCEID'])) {
            foreach ($params as $key => $param) {
                unset($params[$key]);
                $newKey = str_replace('PAYMENTREQUEST_0_', '', $key);
                $params[$newKey] = $param;
            }
            $result = $client->doReferenceTransaction($params);
            $params['PAYMENTREQUEST_0_CUSTOM'] = $params['CUSTOM'];
            $result['PAYMENTINFO_0_TRANSACTIONID'] = $result['TRANSACTIONID'];
            $result['PAYMENTINFO_0_PAYMENTSTATUS'] = $result['PAYMENTSTATUS'];
            $result['PAYMENTINFO_0_AMT'] = $result['AMT'];
        } else {
            $result = $client->doExpressCheckoutPayment($params);
        }
        $result['PAYMENTREQUEST_0_CUSTOM'] = $params['PAYMENTREQUEST_0_CUSTOM'];

        if ($result['ACK'] !== 'Success') {
            return $result;
        }

        if (empty($orderNumber)) {
            $orderNumber = $this->saveOrder(
                $result['PAYMENTINFO_0_TRANSACTIONID'],
                $result['PAYMENTREQUEST_0_CUSTOM']
            );
        }

        // Sets billing agreement id
        if (!empty($result['BILLINGAGREEMENTID'])) {
            try {
                $sql = '
                    INSERT INTO s_order_attributes (orderID, swag_payal_billing_agreement_id)
                    SELECT id, ? FROM s_order WHERE ordernumber = ?
                    ON DUPLICATE KEY UPDATE
                       swag_payal_billing_agreement_id = VALUES(swag_payal_billing_agreement_id)
                ';
                $this->get('db')->query($sql, array($result['BILLINGAGREEMENTID'], $orderNumber));
            } catch (Exception $e) {
            }
        }

        // Sets express flag
        if (!empty($params['TOKEN'])) {
            try {
                $sql = '
                    INSERT INTO s_order_attributes (orderID, swag_payal_express)
                    SELECT id, 1 FROM s_order WHERE ordernumber = ?
                    ON DUPLICATE KEY UPDATE swag_payal_express = 1
                ';
                $this->get('db')->query($sql, array($orderNumber));
            } catch (Exception $e) {
            }
        }

        // Stets transaction details
        $sql = '
            UPDATE `s_order`
            SET transactionID = ?, internalcomment = CONCAT(internalcomment, ?),
              customercomment = CONCAT(customercomment, ?)
            WHERE ordernumber = ?
        ';
        $this->get('db')->query(
            $sql,
            array(
                $result['PAYMENTINFO_0_TRANSACTIONID'],
                isset($details['EMAIL']) ? "{$details['EMAIL']} ({$details['PAYERSTATUS']})\r\n" : null,
                isset($details['NOTE']) ? $details['NOTE'] : '',
                $orderNumber,
            )
        );

        // Sets payment status
        $paymentStatus = $result['PAYMENTINFO_0_PAYMENTSTATUS'];
        $ppAmount = (float) $result['PAYMENTINFO_0_AMT'];
        $swAmount = $this->getAmount();
        if (abs($swAmount - $ppAmount) >= 0.01) {
            $paymentStatus = 'AmountMissMatch'; //Überprüfung notwendig
        }
        $this->plugin->setPaymentStatus($result['PAYMENTINFO_0_TRANSACTIONID'], $paymentStatus);

        $result['PAYMENTREQUEST_0_INVNUM'] = $orderNumber;

        return $result;
    }

    protected function createAccount(array $details)
    {
        /** @var sAdmin $module */
        $module = $this->get('modules')->Admin();
        $session = $this->session;
        $encoderName = null;

        if (!defined('Shopware::VERSION') || Shopware::VERSION === '___VERSION___' || version_compare(Shopware::VERSION, '4.1.0', '>=')) {
            $encoderName = $this->get('passwordEncoder')->getDefaultPasswordEncoderName();
        }

        $data['auth']['email'] = $details['EMAIL'];
        $data['auth']['password'] = $details['PAYERID'];
        $data['auth']['accountmode'] = '1';

        $data['billing']['salutation'] = 'mr';
        $possibleSalutations = $this->get('config')->get('shopsalutations');
        if ($possibleSalutations !== null) {
            // set different salutation is only possible in SW >= 5.2.0
            $possibleSalutations = explode(',', $possibleSalutations);
            // as PayPal does not provide a salutation, we have to set one of the possible options
            $data['billing']['salutation'] = isset($possibleSalutations[0]) ? $possibleSalutations[0] : 'mr';
        }

        $data['billing']['firstname'] = $details['FIRSTNAME'];
        $data['billing']['lastname'] = $details['LASTNAME'];

        if (defined('Shopware::VERSION')) {
            $swVersion = Shopware::VERSION;
        } else {
            $swVersion = Shopware()->Container()->get('config')->get('version');
        }

        if (version_compare($swVersion, '4.4.0', '>=') && version_compare($swVersion, '5.2.0', '<')) {
            $data['billing']['street'] = $details['PAYMENTREQUEST_0_SHIPTOSTREET'];
            if (!empty($details['PAYMENTREQUEST_0_SHIPTOSTREET2'])) {
                $data['billing']['additional_address_line1'] = $details['PAYMENTREQUEST_0_SHIPTOSTREET2'];
            }
        } elseif ($swVersion === '___VERSION___' || version_compare($swVersion, '5.2.0', '>=')) {
            $data['billing']['street'] = $details['PAYMENTREQUEST_0_SHIPTOSTREET'];
            if (!empty($details['PAYMENTREQUEST_0_SHIPTOSTREET2'])) {
                $data['billing']['additionalAddressLine1'] = $details['PAYMENTREQUEST_0_SHIPTOSTREET2'];
            }
        } else {
            $street = explode(' ', $details['PAYMENTREQUEST_0_SHIPTOSTREET']);
            $data['billing']['street'] = $street[0];
            $data['billing']['streetnumber'] = implode(' ', array_slice($street, 1));
            if (strlen($data['billing']['streetnumber']) > 4) {
                $data['billing']['street'] .= ' ' . $data['billing']['streetnumber'];
                $data['billing']['streetnumber'] = '';
            }
            if (empty($data['billing']['streetnumber'])) {
                $data['billing']['streetnumber'] = ' ';
            }
        }

        $data['billing']['zipcode'] = $details['PAYMENTREQUEST_0_SHIPTOZIP'];
        $data['billing']['city'] = $details['PAYMENTREQUEST_0_SHIPTOCITY'];
        $sql = 'SELECT id FROM s_core_countries WHERE countryiso=?';
        $countryId = $this->get('db')->fetchOne($sql, array($details['PAYMENTREQUEST_0_SHIPTOCOUNTRYCODE']));
        $data['billing']['country'] = $countryId;
        if (!empty($details['PAYMENTREQUEST_0_SHIPTOSTATE']) && $details['PAYMENTREQUEST_0_SHIPTOSTATE'] !== 'Empty') {
            $sql = 'SELECT id FROM s_core_countries_states WHERE countryID=? AND shortcode=?';
            $stateId = $this->get('db')->fetchOne($sql, array($countryId, $details['PAYMENTREQUEST_0_SHIPTOSTATE']));
            $data['billing']['stateID'] = $stateId;
        }
        if (!empty($details['BUSINESS'])) {
            $data['billing']['customer_type'] = 'business';
            $data['billing']['company'] = $details['BUSINESS'];
        } else {
            $data['billing']['customer_type'] = 'private';
            $data['billing']['company'] = '';
        }
        $data['billing']['department'] = '';

        $data['shipping'] = $data['billing'];
        $name = explode(' ', $details['PAYMENTREQUEST_0_SHIPTONAME']);
        $data['shipping']['firstname'] = $name[0];
        $data['shipping']['lastname'] = implode(' ', array_slice($name, 1));
        if (!empty($details['PAYMENTREQUEST_0_SHIPTOPHONENUM'])) {
            $data['billing']['phone'] = $details['PAYMENTREQUEST_0_SHIPTOPHONENUM'];
        } elseif (!empty($details['PHONENUM'])) {
            $data['billing']['phone'] = $details['PHONENUM'];
        }
        if (!empty($data['billing']['phone'])) {
            $data['shipping']['phone'] = $data['billing']['phone'];
        }

        $sql = 'SELECT id FROM s_core_paymentmeans WHERE name=?';
        $paymentId = $this->get('db')->fetchOne($sql, array('paypal'));
        $data['payment']['object'] = $module->sGetPaymentMeanById($paymentId);

        $shop = $this->get('shop');
        $shop = $shop->getMain() ?: $shop;
        $sql = 'SELECT `password` FROM `s_user` WHERE `email` LIKE ? AND `active` = 1 ';
        if ($shop->getCustomerScope()) {
            $sql .= "AND `subshopID` = {$shop->getId()} ";
        }

        //Always use the latest account. It is possible, that the account already exists but the password may be invalid.
        //The plugin then creates a new account and uses that one instead.
        $sql .= 'ORDER BY `id` DESC';
        $data['auth']['passwordMD5'] = $this->get('db')->fetchOne($sql, array($data['auth']['email']));

        // First try login / Reuse paypal account
        $module->sSYSTEM->_POST = $data['auth'];
        $module->sLogin(true);

        // Check login status
        if ($module->sCheckUser()) {
            //Save the new address.
            if ($swVersion === '___VERSION___' || version_compare($swVersion, '5.2.0', '>=')) {
                $userId = $this->session->offsetGet('sUserId');
                $this->updateShipping($userId, $data['shipping']);
            } else {
                $module->sSYSTEM->_POST = $data['shipping'];
                $module->sUpdateShipping();
            }

            $module->sSYSTEM->_POST = array('sPayment' => $paymentId);
            $module->sUpdatePayment();
        } else {
            if ($encoderName !== null) {
                $data['auth']['encoderName'] = $encoderName;
                $data['auth']['password'] = $this->get('passwordEncoder')
                    ->encodePassword($data['auth']['password'], $encoderName);
            } else {
                $data['auth']['password'] = md5($data['auth']['password']);
            }
            $session->offsetSet('sRegisterFinished', false);
            if (version_compare($swVersion, '4.3.0', '>=') && version_compare($swVersion, '5.2.0', '<')) {
                $session->offsetSet('sRegister', $data);
            } elseif (version_compare($swVersion, '4.3.0', '<')) {
                $session->offsetSet('sRegister', new ArrayObject($data, ArrayObject::ARRAY_AS_PROPS));
            }

            if ($swVersion === '___VERSION___' || version_compare($swVersion, '5.2.0', '>=')) {
                $this->saveUser($data);
                $module->sSYSTEM->_POST = $data['auth'];
                $module->sLogin(true);
                $this->returnAction();
            } else {
                $module->sSaveRegister();
            }
        }
    }

    /**
     * Returns the article list parameter data.
     *
     * @return array
     */
    protected function getBasketParameter()
    {
        $params = array();
        $user = $this->getUser();

        $params['PAYMENTREQUEST_0_CURRENCYCODE'] = $this->getCurrencyShortName();

        if ($user !== null) {
            $basket = $this->getBasket();
            if (!empty($basket['sShippingcosts'])) {
                $params['PAYMENTREQUEST_0_SHIPPINGAMT'] = $this->getShipment();
            }
            $params['PAYMENTREQUEST_0_AMT'] = $this->getAmount();
        } else {
            $basket = $this->get('modules')->Basket()->sGetBasket();
            if (!empty($basket['sShippingcosts'])) {
                $params['PAYMENTREQUEST_0_SHIPPINGAMT'] = !empty($basket['sShippingcostsWithTax']) ? $basket['sShippingcostsWithTax'] : $basket['sShippingcosts'];
                $params['PAYMENTREQUEST_0_SHIPPINGAMT'] = str_replace(',', '.', $params['PAYMENTREQUEST_0_SHIPPINGAMT']);
            }
            if (!empty($user['additional']['charge_vat']) && !empty($item['AmountWithTaxNumeric'])) {
                $params['PAYMENTREQUEST_0_AMT'] = $basket['AmountWithTaxNumeric'];
            } else {
                $params['PAYMENTREQUEST_0_AMT'] = $basket['AmountNumeric'];
            }
            $params['PAYMENTREQUEST_0_AMT'] = $basket['AmountNumeric'];
        }
        $params['PAYMENTREQUEST_0_AMT'] = number_format($params['PAYMENTREQUEST_0_AMT'], 2, '.', '');
        $params['PAYMENTREQUEST_0_SHIPPINGAMT'] = number_format($params['PAYMENTREQUEST_0_SHIPPINGAMT'], 2, '.', '');
        $params['PAYMENTREQUEST_0_ITEMAMT'] = number_format($params['PAYMENTREQUEST_0_AMT'] - $params['PAYMENTREQUEST_0_SHIPPINGAMT'], 2, '.', '');
        $params['PAYMENTREQUEST_0_TAXAMT'] = number_format(0, 2, '.', '');

        $config = $this->plugin->Config();
        if ($config->get('paypalTransferCart') && $params['PAYMENTREQUEST_0_ITEMAMT'] !== '0.00' && count($basket['content']) < 25) {
            $key = 0;
            $lastCustomProduct = null;
            foreach ($basket['content'] as $basketItem) {
                $sku = $basketItem['ordernumber'];
                $name = $basketItem['articlename'];
                $quantity = (int) $basketItem['quantity'];
                if (!empty($user['additional']['charge_vat']) && !empty($basketItem['amountWithTax'])) {
                    $amount = round($basketItem['amountWithTax'], 2);
                } else {
                    $amount = str_replace(',', '.', $basketItem['amount']);
                }

                // If more than 2 decimal places
                if (round($amount / $quantity, 2) * $quantity != $amount) {
                    if ($quantity !== 1) {
                        $name = $quantity . 'x ' . $name;
                    }
                    $quantity = 1;
                } else {
                    $amount = round($amount / $quantity, 2);
                }

                // Add support for custom products
                if (!empty($basketItem['customProductMode'])) {
                    switch ($basketItem['customProductMode']) {
                        case 1: // Product
                            $lastCustomProduct = $key;
                            break;
                        case 2: // Option
                            $lastCustomProductNumber = 'L_PAYMENTREQUEST_0_NUMBER' . $lastCustomProduct;
                            if (empty($sku) && isset($params[$lastCustomProductNumber])) {
                                $sku = $params['L_PAYMENTREQUEST_0_NUMBER' . $lastCustomProduct];
                            }
                            break;
                        case 3: // Value
                            $last = $key - 1;
                            $lastNumber = 'L_PAYMENTREQUEST_0_NAME' . $last;
                            if (isset($params[$lastNumber])) {
                                if (strpos($params['L_PAYMENTREQUEST_0_NAME' . $last], ': ') === false) {
                                    $params['L_PAYMENTREQUEST_0_NAME' . $last] .= ': ' . $name;
                                } else {
                                    $params['L_PAYMENTREQUEST_0_NAME' . $last] .= ', ' . $name;
                                }
                                $params['L_PAYMENTREQUEST_0_AMT' . $last] += $amount;
                            }
                            continue 2;
                        default:
                            break;
                    }
                }

                $article = array(
                    'L_PAYMENTREQUEST_0_NUMBER' . $key => $sku,
                    'L_PAYMENTREQUEST_0_NAME' . $key => $name,
                    'L_PAYMENTREQUEST_0_AMT' . $key => $amount,
                    'L_PAYMENTREQUEST_0_QTY' . $key => $quantity,
                );
                $params = array_merge($params, $article);
                ++$key;
            }
        }

        if ($params['PAYMENTREQUEST_0_ITEMAMT'] === '0.00') {
            $params['PAYMENTREQUEST_0_ITEMAMT'] = $params['PAYMENTREQUEST_0_SHIPPINGAMT'];
            $params['PAYMENTREQUEST_0_SHIPPINGAMT'] = '0.00';
        }

        return $params;
    }

    /**
     * Returns the prepared customer parameter data.
     *
     * @return array
     */
    protected function getCustomerParameter()
    {
        $user = $this->getUser();
        if (empty($user)) {
            return array(
                'LOCALECODE' => $this->plugin->getLocaleCode(true),
            );
        }

        $shipping = $user['shippingaddress'];
        $name = $shipping['firstname'] . ' ' . $shipping['lastname'];
        if (!empty($shipping['company'])) {
            $name = $shipping['company'] . ' - ' . $name;
        }

        if (!empty($shipping['streetnumber'])) {
            $shipping['street'] .= ' ' . $shipping['streetnumber'];
        }

        if (defined('Shopware::VERSION')) {
            $swVersion = Shopware::VERSION;
        } else {
            $swVersion = Shopware()->Container()->get('config')->get('version');
        }

        if (version_compare($swVersion, '4.4.0', '>=') && version_compare($swVersion, '5.2.0', '<')) {
            if (!empty($shipping['additional_address_line1'])) {
                $shipping['street2'] = $shipping['additional_address_line1'];
                if (!empty($shipping['additional_address_line2'])) {
                    $shipping['street2'] .= ' ' . $shipping['additional_address_line2'];
                }
            }
        } elseif ($swVersion === '___VERSION___' || version_compare($swVersion, '5.2.0', '>=')) {
            if (!empty($shipping['additionalAddressLine1'])) {
                $shipping['street2'] = $shipping['additionalAddressLine1'];
                if (!empty($shipping['additionalAddressLine2'])) {
                    $shipping['street2'] .= ' ' . $shipping['additionalAddressLine2'];
                }
            }
        } else {
            $shipping['street2'] = '';
        }

        $customer = array(
            'CUSTOMERSERVICENUMBER' => $user['billingaddress']['customernumber'],
            //'gender' => $shipping['salutation'] == 'ms' ? 'f' : 'm',
            'PAYMENTREQUEST_0_SHIPTONAME' => $name,
            'PAYMENTREQUEST_0_SHIPTOSTREET' => $shipping['street'],
            'PAYMENTREQUEST_0_SHIPTOSTREET2' => $shipping['street2'],
            'PAYMENTREQUEST_0_SHIPTOZIP' => $shipping['zipcode'],
            'PAYMENTREQUEST_0_SHIPTOCITY' => $shipping['city'],
            'PAYMENTREQUEST_0_SHIPTOCOUNTRYCODE' => $user['additional']['countryShipping']['countryiso'],
            'EMAIL' => $user['additional']['user']['email'],
            'PAYMENTREQUEST_0_SHIPTOPHONENUM' => $user['billingaddress']['phone'],
            'LOCALECODE' => $this->plugin->getLocaleCode(true),
        );

        if (!empty($user['additional']['stateShipping']['shortcode'])) {
            $customer['PAYMENTREQUEST_0_SHIPTOSTATE'] = $user['additional']['stateShipping']['shortcode'];
        }

        return $customer;
    }

    /**
     * Saves a new user to the system.
     *
     * @param array $data
     */
    private function saveUser($data)
    {
        $plain = array_merge($data['auth'], $data['shipping']);

        //Create forms and validate the input
        $customer = new Shopware\Models\Customer\Customer();
        $form = $this->createForm('Shopware\Bundle\AccountBundle\Form\Account\PersonalFormType', $customer);
        $form->submit($plain);

        $address = new Shopware\Models\Customer\Address();
        $form = $this->createForm('Shopware\Bundle\AccountBundle\Form\Account\AddressFormType', $address);
        $form->submit($plain);

        /** @var Shopware\Bundle\StoreFrontBundle\Struct\ShopContextInterface $context */
        $context = $this->get('shopware_storefront.context_service')->getShopContext();

        /** @var Shopware\Bundle\StoreFrontBundle\Struct\Shop $shop */
        $shop = $context->getShop();

        /** @var Shopware\Bundle\AccountBundle\Service\RegisterServiceInterface $registerService */
        $registerService = $this->get('shopware_account.register_service');
        $registerService->register($shop, $customer, $address, $address);
    }

    /**
     * Updates the shipping address to the latest address that has been provided by PayPal.
     *
     * @param int   $userId
     * @param array $shippingData
     */
    private function updateShipping($userId, $shippingData)
    {
        /** @var ModelManager $em */
        $em = $this->get('models');

        /** @var Customer $customer */
        $customer = $em->getRepository('Shopware\Models\Customer\Customer')->findOneBy(array('id' => $userId));

        /** @var Address $address */
        $address = $customer->getDefaultShippingAddress();

        $form = $this->createForm('Shopware\Bundle\AccountBundle\Form\Account\AddressFormType', $address);
        $form->submit($shippingData);

        $this->get('shopware_account.address_service')->update($address);
    }

    /**
     * Helper method to log an error
     *
     * @param array $response
     */
    private function logError($response)
    {
        if (!$this->plugin->isAtLeastShopware42()) {
            return;
        }

        $message = '[' . $response['L_ERRORCODE0'] . '] - ' . $response['L_SHORTMESSAGE0'] . '. ' . $response['L_LONGMESSAGE0'];
        /** @var Logger $pluginLogger */
        $pluginLogger = Shopware()->Container()->get('pluginlogger');
        $pluginLogger->error($message);
    }
}
